/*
 * Copyright (C) 2000-2025 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <sys/stat.h>

#include "common.h"
#include "xine-toolkit/backend.h"
#include "xine-toolkit/button.h"
#include "xine-toolkit/combo.h"
#include "xine-toolkit/font.h"
#include "xine-toolkit/image.h"
#include "xine-toolkit/intbox.h"
#include "xine-toolkit/labelbutton.h"
#include "xine-toolkit/inputtext.h"
#include "xine-toolkit/label.h"
#include "xine-toolkit/slider.h"
#include "xine-toolkit/tabs.h"
#include "xine-toolkit/window.h"
#include "setup.h"
#include "actions.h"
#include "event.h"
#include "file_browser.h"

#include <xine/sorted_array.h>
/** TEST */
/* #undef XINE_SARRAY_MODE_UNIQUE */

#define WINDOW_WIDTH  630
#define WINDOW_HEIGHT 530

#define FRAME_HEIGHT   44

#define MAX_DISPLAY_WIDGETS      8

typedef union {
  char s[4];
  uint32_t v;
} _setup_string4_t;

typedef enum {
  _T_frame = 0,
  _T_label,
  _T_widget,
  _T_browse,
  _T_default,
  _T_LAST
} _T_t;

typedef struct {
  xitk_widget_t    *w[_T_LAST];
  struct {
    uint16_t        x, y;
  }                 r[_T_LAST];
  int               changed;
  /* cfg.unknown_value is only used by cfg.type == XINE_CONFIG_TYPE_UNKNOWN,
   * which we skip. misuse it for instance ptr in our copy. */
  xine_cfg_entry_t  cfg;
} _widget_triplet_t;

typedef enum {
  _S_tabs = 0,
  _S_apply,
  _S_ok,
  _S_close,
  _S_slider,
  _S_LAST
} _s_t;

struct xui_setup_st {
  gui_new_window_t      nw;

  xitk_widget_t        *w[_S_LAST];

  struct {
#define SETUP_MAX_SECTIONS 24
    const char         *name[SETUP_MAX_SECTIONS + 1]; /** actually a const _setup_string_t *. */
    uint16_t            nlen[SETUP_MAX_SECTIONS + 1];
    int                 num[SETUP_MAX_SECTIONS];
    int                 first_shown[SETUP_MAX_SECTIONS];
    uint16_t            have, shown;
  }                     sections;

  int                    (*exit) (xui_setup_t *setup);

  _widget_triplet_t    *wg;
  int                   num_wg;
  int                   max_wg;
  int                   first_displayed;

  filebrowser_t        *browser;

  xitk_register_key_t   dialog;

  uint16_t              th, fh; /* Tabs height, Font height */

  xitk_rect_t           main, frame, sl;

  _setup_string4_t      namebuf[320];
};

/*
 * Leaving setup panel, release memory.
 */
static int _setup_exit (xui_setup_t *setup) {
  filebrowser_end (setup->browser);
  setup->browser = NULL;

  xitk_unregister_event_handler (setup->nw.gui->xitk, &setup->dialog);
  gui_window_delete (&setup->nw);

  setup->nw.gui->setup = NULL;
  free (setup->wg);
  free (setup);
  return 1;
}

/*
 *
 */
static void setup_paint_widgets (xui_setup_t *setup, int first) {
  int i, last;
  int y = setup->frame.y;

  last = setup->num_wg - setup->first_displayed;
  if (last > MAX_DISPLAY_WIDGETS)
    last = MAX_DISPLAY_WIDGETS;
  last += setup->first_displayed;

  /* First, disable old widgets. */
  for (i = setup->first_displayed; i < last; i++)
    xitk_widgets_state (setup->wg[i].w, _T_LAST, XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE, 0);

  last = setup->num_wg - MAX_DISPLAY_WIDGETS;
  if (first > last)
    first = last;
  if (first < 0)
    first = 0;
  setup->first_displayed = first;
  last = setup->num_wg - setup->first_displayed;
  if (last > MAX_DISPLAY_WIDGETS)
    last = MAX_DISPLAY_WIDGETS;
  last += setup->first_displayed;

  /* Move widgets to new position. */
  for (i = setup->first_displayed; i < last; i++) {
    _widget_triplet_t *t = setup->wg + i;

    if (t->w[_T_frame])
      xitk_set_widget_pos (t->w[_T_frame], t->r[_T_frame].x, y + t->r[_T_frame].y);
    y += 3;
    if (t->w[_T_label])
      xitk_set_widget_pos (t->w[_T_label], t->r[_T_label].x, y + t->r[_T_label].y);
    if (t->w[_T_widget])
      xitk_set_widget_pos (t->w[_T_widget], t->r[_T_widget].x, y + t->r[_T_widget].y);
    if (t->w[_T_browse])
      xitk_set_widget_pos (t->w[_T_browse], t->r[_T_browse].x, y + t->r[_T_browse].y);
    if (t->w[_T_default])
      xitk_set_widget_pos (t->w[_T_default], t->r[_T_default].x, y + t->r[_T_default].y);
    y += setup->frame.height;
  }

  /* Repaint our new ones just 1 time. */
  if (setup->num_wg > MAX_DISPLAY_WIDGETS)
    xitk_widgets_state (&setup->w[_S_slider], 1, XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE, ~0u);
  for (i = setup->first_displayed; i < last; i++)
    xitk_widgets_state (setup->wg[i].w, _T_LAST, XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE, ~0u);
}

/*
 * Handle X events here.
 */

static int setup_event (void *data, const xitk_be_event_t *e) {
  static const uint8_t bk[16] = {
    [XITK_BUTTON_WHEEL_UP]   = XITK_MOUSE_WHEEL_UP,
    [XITK_BUTTON_WHEEL_DOWN] = XITK_MOUSE_WHEEL_DOWN,
    [XITK_BUTTON_SIDE_LEFT]  = XITK_KEY_LEFT,
    [XITK_BUTTON_SIDE_RIGHT] = XITK_KEY_RIGHT
  };
  xui_setup_t *setup = data;
  xitk_be_event_t fake;
  char s[4];

  switch (e->type) {
    case XITK_EV_DEL_WIN:
      return setup->exit (setup);
    case XITK_EV_BUTTON_DOWN:
    case XITK_EV_BUTTON_UP:
      if (!(s[1] = bk[e->code & 15]))
        break;
      fake = *e;
      fake.utf8 = s;
      fake.type = (e->type == XITK_EV_BUTTON_DOWN) ? XITK_EV_KEY_DOWN : XITK_EV_KEY_UP;
      s[0] = XITK_CTRL_KEY_PREFIX;
      s[2] = 0;
      e = &fake;
      /* fall through */
    case XITK_EV_KEY_DOWN:
    case XITK_EV_KEY_UP:
      if (e->utf8[0] == XITK_CTRL_KEY_PREFIX) {
        switch (e->utf8[1]) {
          case XITK_KEY_UP:
          case XITK_KEY_DOWN:
          case XITK_KEY_NEXT:
          case XITK_KEY_PREV:
          case XITK_KEY_HOME:
          case XITK_KEY_END:
          case XITK_MOUSE_WHEEL_UP:
          case XITK_MOUSE_WHEEL_DOWN:
            return xitk_widget_key_event (setup->w[_S_slider], e, 2) | 1;
          case XITK_KEY_LEFT:
          case XITK_KEY_RIGHT:
            return xitk_widget_key_event (setup->w[_S_tabs], e, 2) | 1;
          case XITK_KEY_RETURN:
            if (!setup->wg || (setup->num_wg <= 0))
              break;
            if (e->type == XITK_EV_KEY_UP)
              xitk_set_focus_to_widget (setup->wg[setup->first_displayed].w[_T_widget]);
            return 1;
          case XITK_KEY_ESCAPE:
            return xitk_widget_key_event (setup->w[_S_close], e, 1);
          default: ;
        }
      } else if (e->utf8[0] == (0x1f & 's')) {
        return xitk_widget_key_event (setup->w[_S_apply], e, 1);
      }
      break;
    default: ;
  }
  return gui_handle_be_event (setup->nw.gui, e);
}

static void any_update (_widget_triplet_t *wt) {
  xui_setup_t *setup = (xui_setup_t *)wt->cfg.unknown_value;
  wt->changed = 1;
  xitk_widgets_state (&setup->w[_S_apply], 1, XITK_WIDGET_STATE_ENABLE, ~0u);
}

static void numtype_update (xitk_widget_t *w, void *data, int value, unsigned int modifier) {
  _widget_triplet_t *triplet = (_widget_triplet_t *) data;

  (void)w;
  (void)modifier;
  /* intentionally always mark as changed, and allow the user
   * to trigger update with same value. */
  triplet->cfg.num_value = value;
  xitk_button_set_state (triplet->w[_T_default], value == triplet->cfg.num_default);
  any_update (triplet);
}

static void default_update (xitk_widget_t *w, void *data, int value, unsigned int modifier) {
  _widget_triplet_t *triplet = (_widget_triplet_t *) data;

  (void)modifier;
  if (!value) {
    /* unset the default state makes not much sense. */
    xitk_button_set_state (w, 1);
  } else {
    switch (triplet->cfg.type) {
      case XINE_CONFIG_TYPE_STRING:
        xitk_inputtext_change_text (triplet->w[_T_widget], triplet->cfg.str_default);
        break;
      case XINE_CONFIG_TYPE_RANGE:
      case XINE_CONFIG_TYPE_NUM:
        triplet->cfg.num_value = triplet->cfg.num_default;
        xitk_intbox_set_value (triplet->w[_T_widget], triplet->cfg.num_default);
        break;
      case XINE_CONFIG_TYPE_ENUM:
        triplet->cfg.num_value = triplet->cfg.num_default;
        xitk_combo_set_select (triplet->w[_T_widget], triplet->cfg.num_default);
        break;
      case XINE_CONFIG_TYPE_BOOL:
        triplet->cfg.num_value = triplet->cfg.num_default;
        xitk_button_set_state (triplet->w[_T_widget], triplet->cfg.num_default);
        break;
      default: ;
    }
    any_update (triplet);
  }
}

static void stringtype_update(xitk_widget_t *w, void *data, const char *str) {
  _widget_triplet_t *triplet = (_widget_triplet_t *) data;

  (void)w;
  (void)str;
  xitk_button_set_state (triplet->w[_T_default], triplet->cfg.str_default && !strcmp (str, triplet->cfg.str_default));
  any_update (triplet);
}

static void _browse_select_callback (filebrowser_t *fb, void *data) {
  _widget_triplet_t *triplet = (_widget_triplet_t *) data;
  xui_setup_t *setup = (xui_setup_t *)triplet->cfg.unknown_value;
  char buf[XITK_PATH_MAX + XITK_NAME_MAX + 2];
  size_t slen;

  if ((slen = filebrowser_get_string (fb, buf + 1, sizeof (buf) - 2, 1))) {
    if (triplet->cfg.num_value == XINE_CONFIG_STRING_IS_DIRECTORY_NAME) {
      struct stat sbuf;
      /* remove trailing slash except for "/". */
      buf[0] = 0;
      if ((buf[1 + slen - 1] == '/') && (slen > 1)) {
        buf[slen--] = 0;
      }
      /* we want a directory. if we got a file or device, use its parent. */
      else if (!stat (buf + 1, &sbuf) &&
        (S_ISREG (sbuf.st_mode) || S_ISCHR (sbuf.st_mode) || S_ISBLK (sbuf.st_mode) || S_ISFIFO (sbuf.st_mode))) {
        char *p;

        buf[0] = '/';
        for (p = buf + slen; *p != '/'; p--) ;
        /* keep "/" */
        if (p == buf + 1)
          p++;
        *p = 0;
      }
    }
    xitk_inputtext_change_text (triplet->w[_T_widget], buf + 1);
    xitk_button_set_state (triplet->w[_T_default], triplet->cfg.str_default && !strcmp (buf + 1, triplet->cfg.str_default));
    any_update (triplet);
  }
  setup->browser = NULL;
}

static void _browse_exit_callback (filebrowser_t *fb, void *data) {
  _widget_triplet_t *triplet = (_widget_triplet_t *) data;
  xui_setup_t *setup = (xui_setup_t *)triplet->cfg.unknown_value;

  (void)fb;
  setup->browser = NULL;
}

static void open_browser (xitk_widget_t *w, void *data, int state, unsigned int modifier) {
  _widget_triplet_t *triplet = (_widget_triplet_t *) data;
  xui_setup_t *setup = (xui_setup_t *)triplet->cfg.unknown_value;
  filebrowser_callback_button_t  cbb[2] = {
    [0] = {
      .label = _("Select"),
      .callback = _browse_select_callback,
      .userdata = triplet,
      .need_a_file = triplet->cfg.num_value != XINE_CONFIG_STRING_IS_DIRECTORY_NAME
    },
    [1] = {
      .callback = _browse_exit_callback,
      .userdata = triplet
    }
  };

  (void)w;
  (void)state;
  (void)modifier;
  /* NULL safe. */
  filebrowser_end (setup->browser);
  setup->browser = filebrowser_create (setup->nw.gui, setup->nw.xwin,
    triplet->cfg.description, triplet->cfg.str_value, cbb, 2, XUI_EXTS_ANY);
}

/*
 *
 */
static void setup_section_widgets (xui_setup_t *setup, int s) {
  int cfg_err_result, img_refs = 0;
  const unsigned int known_types = (1 << XINE_CONFIG_TYPE_RANGE)
                                 | (1 << XINE_CONFIG_TYPE_STRING)
                                 | (1 << XINE_CONFIG_TYPE_ENUM)
                                 | (1 << XINE_CONFIG_TYPE_NUM)
                                 | (1 << XINE_CONFIG_TYPE_BOOL);
  xine_cfg_entry_t entry;
  xitk_image_widget_t im = { .nw = { .wl = setup->nw.wl, .add_state = XITK_WIDGET_STATE_CLEAR } };
  xitk_label_widget_t lb = { .nw = { .wl = setup->nw.wl, .add_state = XITK_WIDGET_STATE_CLEAR } };
  xitk_intbox_widget_t ib = {
    .nw = { .wl = setup->nw.wl, .add_state = XITK_WIDGET_STATE_CLEAR },
    .callback = numtype_update
  };
  xitk_inputtext_widget_t inp = {
    .nw = { .wl = setup->nw.wl, .add_state = XITK_WIDGET_STATE_CLEAR },
    .callback = stringtype_update
  };
  xitk_combo_widget_t cmb = {
    .nw = { .wl = setup->nw.wl, .add_state = XITK_WIDGET_STATE_CLEAR },
    .layer_above = gui_layer_above (setup->nw.gui, NULL),
    .parent_wkey = &setup->nw.key,
    .callback    = numtype_update
  };
  xitk_button_widget_t b = {
    .nw = { .wl = setup->nw.wl, .add_state = XITK_WIDGET_STATE_CLEAR },
    .symbol = XITK_SYMBOL_CHECK,
    .state_callback = numtype_update
  };
  xitk_labelbutton_widget_t br = {
    .nw = {
        .wl = setup->nw.wl,
        .add_state = XITK_WIDGET_STATE_CLEAR,
        .tips = xitk_ref_string_ref (_("Select"), 0) /* << make shared string */
    },
    .button_type = CLICK_BUTTON,
    .align = ALIGN_CENTER,
    .label = "...",
    .callback = open_browser
  };
  xitk_button_widget_t d = {
    .nw = {
        .wl = setup->nw.wl,
        .add_state = XITK_WIDGET_STATE_CLEAR,
        .tips = xitk_ref_string_ref (_("Default value"), 0) /* << make shared string */
    },
    .symbol = XITK_SYMBOL_LEFT,
    .state_callback = default_update
  };
#define SHARED_FRAME_SIZE 8
  xitk_part_image_t shared_frame = {
    .image = NULL,
    .x = 0,
    .width = setup->frame.width,
    .height = setup->frame.height,
    .num_states = 1
  };
  int shared_index = SHARED_FRAME_SIZE - 1;
  const uint32_t style[2] = {
    XITK_DRAW_INNER | XITK_DRAW_R | XITK_DRAW_G | (XITK_DRAW_SAT (setup->nw.gui->gfx_saturation)),
    XITK_DRAW_INNER | XITK_DRAW_G | XITK_DRAW_B | (XITK_DRAW_SAT (setup->nw.gui->gfx_saturation))
  };
  char sname[200];
  size_t nlen = setup->sections.nlen[s] <= sizeof (sname) - 2 ? setup->sections.nlen[s] : sizeof (sname) - 2;

  memcpy (sname, setup->sections.name[s], nlen);
  memcpy (sname + nlen, ".", 2);
  nlen++;

  xitk_widgets_state (&setup->w[_S_slider], 1, XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE, 0);
  xitk_widgets_state (&setup->w[_S_apply], 1, XITK_WIDGET_STATE_ENABLE, 0);

  setup->sections.shown = s;
  memset (&entry, 0, sizeof (entry));

  cfg_err_result = setup->wg ? xine_config_get_first_entry (setup->nw.gui->xine, &entry) : 0;

  while (cfg_err_result) {

    if ((entry.exp_level <= setup->nw.gui->experience_level)
      && !strncmp (entry.key, sname, nlen)
      && entry.description
      && ((1 << entry.type) & known_types)) {
      _widget_triplet_t   *wt = setup->wg + setup->num_wg;
      const char          *labelkey = entry.key + setup->sections.nlen[s];
      const char          *help = entry.help ? entry.help : (const char *)_("No help available");
      int                  x = setup->frame.x, w1, w2, h;
      uint32_t             bgcolor = 0;

      if (setup->num_wg >= setup->max_wg) {
        /* should be rare. */
        wt = realloc (setup->wg, (setup->max_wg + 8) * sizeof (*setup->wg));
        if (!wt)
          break;
        setup->max_wg += 8;
        setup->wg = wt;
        wt += setup->num_wg;
      }
      setup->num_wg += 1;

      wt->cfg = entry;
      wt->cfg.unknown_value = (char *)setup;
      wt->changed = 0;
      wt->w[_T_browse] = NULL;

      wt->r[_T_frame].x = x;
      wt->r[_T_frame].y = 0;
      if (++shared_index >= SHARED_FRAME_SIZE) {
        shared_index = 0;
        img_refs += xitk_image_free_image (&shared_frame.image);
        shared_frame.image = xitk_image_new (setup->nw.gui->xitk, NULL, 0, setup->frame.width, setup->frame.height * SHARED_FRAME_SIZE);
        if (shared_frame.image)
          xitk_image_fill_rectangle (shared_frame.image, 0, 0, setup->frame.width, setup->frame.height * SHARED_FRAME_SIZE,
            xitk_get_cfg_num (setup->nw.gui->xitk, XITK_BG_COLOR));
      }
      wt->w[_T_frame] = NULL;
      if (shared_frame.image) {
        shared_frame.y = shared_index * setup->frame.height;
        bgcolor = xitk_image_draw_frame (shared_frame.image, entry.description, hboldfontname,
          shared_frame.x, shared_frame.y, setup->frame.width, setup->frame.height, style[shared_index & 1]);
        wt->w[_T_frame] = xitk_noskin_part_image_create (&im, &shared_frame, x, 0);
      }
      x += 10;
      wt->r[_T_widget].x = x;

      switch (entry.type) {

        case XINE_CONFIG_TYPE_RANGE: /* slider */
          ib.nw.userdata = wt;
          /* HACK for stuff like the xv color key. */
          if ((entry.range_min == 0) && (entry.range_max > 0x7fffffff / 260)) {
            ib.fmt = INTBOX_FMT_HASH;
            w1 = 100;
          } else {
            ib.fmt = INTBOX_FMT_DECIMAL;
            w1 = 260;
          }
          ib.nw.group = wt->w[_T_frame];
          ib.min      = entry.range_min;
          ib.max      = entry.range_max;
          ib.value    = entry.num_value;
          ib.step     = 1;
          ib.nw.tips  = help;
          h = 20;
          wt->r[_T_widget].y = (setup->frame.height - h) >> 1;
          wt->w[_T_widget] = xitk_noskin_intbox_create (&ib, x, wt->r[_T_widget].y, w1, h);
          x += w1;
          goto num_default;

        case XINE_CONFIG_TYPE_STRING:
          inp.nw.group = wt->w[_T_frame];
          inp.nw.userdata = wt;
          inp.text       = entry.str_value;
          inp.max_length = 2048;
          inp.nw.tips    = help;
          w2 = 0;
          switch (entry.num_value) {
            case XINE_CONFIG_STRING_IS_FILENAME:
            case XINE_CONFIG_STRING_IS_DEVICE_NAME:
            case XINE_CONFIG_STRING_IS_DIRECTORY_NAME:
              inp.max_length = XITK_PATH_MAX;
              br.nw.group = wt->w[_T_frame];
              br.nw.userdata = wt;
              w1 = 18;
              w2 = 18 + 10;
              h = 18;
              wt->r[_T_browse].x = x + 260 + 10;
              wt->r[_T_browse].y = (setup->frame.height - h) >> 1;
              wt->w[_T_browse] = xitk_noskin_labelbutton_create (&br, wt->r[_T_browse].x, wt->r[_T_browse].y, w1, h,
                XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_INV, tabsfontname);
              break;
            default: ;
          }
          w1 = 260;
          h = 20;
          wt->r[_T_widget].y = (setup->frame.height - h) >> 1;
          wt->w[_T_widget] = xitk_noskin_inputtext_create (&inp, wt->r[_T_widget].x, wt->r[_T_widget].y,
            w1, h, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, fontname);
          x += w1 + w2;
          if (entry.str_default) {
            d.nw.group = wt->w[_T_frame];
            d.nw.userdata = wt;
            w1 = 18;
            h = 18;
            wt->r[_T_default].x = setup->frame.x + setup->frame.width - 18 - 10;
            wt->r[_T_default].y = (setup->frame.height - h) >> 1;
            d.nw.add_state = !strcmp (entry.str_value, entry.str_default) ? XITK_WIDGET_STATE_ON : XITK_WIDGET_STATE_CLEAR;
            wt->w[_T_default] = xitk_noskin_button_create (&d, wt->r[_T_default].x, wt->r[_T_default].y, w1, h);
          } else {
            wt->w[_T_default] = NULL;
          }
          break;

        case XINE_CONFIG_TYPE_ENUM:
          cmb.nw.group = wt->w[_T_frame];
          cmb.nw.userdata = wt;
          cmb.entries     = (const char **)entry.enum_values;
          cmb.nw.tips     = help;
          w1 = 260;
          h = 18;
          wt->r[_T_widget].y = (setup->frame.height - h) >> 1;
          cmb.select = entry.num_value;
          wt->w[_T_widget] = xitk_noskin_combo_create (&cmb, wt->r[_T_widget].x, wt->r[_T_widget].y, w1, h);
          x += w1;
          goto num_default;

        case XINE_CONFIG_TYPE_NUM:
          ib.nw.group = wt->w[_T_frame];
          ib.nw.userdata = wt;
          ib.fmt      = INTBOX_FMT_DECIMAL;
          ib.min      = 0;
          ib.max      = 0;
          ib.value    = entry.num_value;
          ib.step     = 1;
          ib.nw.tips  = help;
          w1 = 70;
          h = 20;
          wt->r[_T_widget].y = (setup->frame.height - h) >> 1;
          wt->w[_T_widget] = xitk_noskin_intbox_create (&ib, wt->r[_T_widget].x, wt->r[_T_widget].y, w1, h);
          x += w1;
          goto num_default;

        case XINE_CONFIG_TYPE_BOOL:
          b.nw.group = wt->w[_T_frame];
          b.nw.userdata = wt;
          b.nw.tips     = help;
          w1 = 14;
          h = 14;
          wt->r[_T_widget].y = (setup->frame.height - h) >> 1;
          b.nw.add_state = entry.num_value ? XITK_WIDGET_STATE_ON : XITK_WIDGET_STATE_CLEAR;
          wt->w[_T_widget] = xitk_noskin_button_create (&b, wt->r[_T_widget].x, wt->r[_T_widget].y, w1, h);
          x += w1;
        num_default:
          d.nw.group = wt->w[_T_frame];
          d.nw.userdata = wt;
          w1 = 18;
          h = 18;
          wt->r[_T_default].x = setup->frame.x + setup->frame.width - 18 - 10;
          wt->r[_T_default].y = (setup->frame.height - h) >> 1;
          d.nw.add_state = (entry.num_value == entry.num_default) ? XITK_WIDGET_STATE_ON : XITK_WIDGET_STATE_CLEAR;
          wt->w[_T_default] = xitk_noskin_button_create (&d, wt->r[_T_default].x, wt->r[_T_default].y, w1, h);
          break;

        default: ;
      }

      {
        char b[1024], *p = b, *e = b + sizeof (b);
        if (!entry.callback_data && !entry.callback) {
          p += strlcpy (p, labelkey, e - p);
          if (p < e)
            strlcpy (p, " (*)", e - p);
          lb.label = b;
        } else {
          lb.label = labelkey;
        }
        lb.nw.group = wt->w[_T_frame];
        lb.nw.tips = help;
        wt->r[_T_label].x = x + 10;
        w1 = setup->frame.x + setup->frame.width - wt->r[_T_label].x - 10 - 20 - 10;
        h = setup->fh;
        wt->r[_T_label].y = (setup->frame.height - h) >> 1;
        wt->w[_T_label] = xitk_noskin_label_color_create (&lb,
          wt->r[_T_label].x, wt->r[_T_label].y, w1, h, fontname, XITK_NOSKIN_TEXT_NORM, bgcolor);
        xitk_widget_set_focus_redirect (wt->w[_T_label], wt->w[_T_widget]);
      }

    }

    memset (&entry, 0, sizeof (entry));
    cfg_err_result = xine_config_get_next_entry (setup->nw.gui->xine, &entry);
  }

  img_refs += xitk_image_free_image (&shared_frame.image);
  setup->sections.num[s] = setup->num_wg;

  {
    int n = xitk_ref_string_unref (&br.nw.tips) + xitk_ref_string_unref (&d.nw.tips);
    if (setup->nw.gui->verbosity >= 2)
      printf ("gui.setup.section (%s): got %d shared string refs, %d shared image refs.\n",
        setup->sections.name[s], n, img_refs);
  }

  if (!setup->num_wg && setup->wg) {
    xitk_image_t *image = xitk_image_from_string (setup->nw.gui->xitk, tabsfontname, setup->frame.width, ALIGN_CENTER,
      _("There is no configuration option available in this user experience level."));
    xitk_image_widget_t im = { .nw = { .wl = setup->nw.wl, .add_state = XITK_WIDGET_STATE_CLEAR } };

    setup->wg->r[_T_frame].x = setup->frame.x;
    setup->wg->r[_T_frame].y = (setup->frame.height * MAX_DISPLAY_WIDGETS - xitk_image_height (image)) >> 1;
    setup->wg->w[_T_frame] = xitk_noskin_image_create (&im, image, setup->wg->r[_T_frame].x, setup->wg->r[_T_frame].y);
    xitk_image_free_image (&image);
    setup->wg->w[_T_widget] = NULL;
    setup->wg->w[_T_label] = NULL;
    setup->wg->w[_T_browse] = NULL;
    setup->wg->w[_T_default] = NULL;
    setup->wg->changed = 0;
    setup->num_wg = 1;
  }

  if (setup->num_wg > MAX_DISPLAY_WIDGETS) {
    xitk_slider_hv_t si = {
      .v = {
        .visible = MAX_DISPLAY_WIDGETS,
        .step = 1,
        .max = setup->num_wg
      }
    };
    xitk_slider_hv_sync (setup->w[_S_slider], &si, XITK_SLIDER_SYNC_SET);
  }
}

/*
 *
 */
static void setup_change_section (xitk_widget_t *wx, void *data, int section, unsigned int modifier) {
  xui_setup_t *setup = data;
  int max;

  (void)wx;
  (void)modifier;
  if (setup->browser) {
    filebrowser_end (setup->browser);
    setup->browser = NULL;
  }
  /* we are so fast now, you will hardly see the watch there ;-) */
  xitk_window_define_window_cursor (setup->nw.xwin, xitk_cursor_watch);

  setup->sections.first_shown[setup->sections.shown] = setup->first_displayed;

  /* remove old widgets */
  {
    _widget_triplet_t *wt;
    for (wt = setup->wg + setup->num_wg - 1; wt >= setup->wg; wt--)
      xitk_widgets_delete (wt->w, _T_LAST);
  }
  setup->num_wg = 0;

  setup_section_widgets (setup, section);
  max = setup->num_wg - MAX_DISPLAY_WIDGETS;
  if (max < 0)
    max = 0;

  /* clear_tab */
  xitk_image_draw_image (setup->nw.wl, setup->nw.bg,
    setup->main.x, setup->main.y, setup->main.width, setup->main.height, setup->main.x, setup->main.y, 0);

  setup_paint_widgets (setup, setup->sections.first_shown[section]);
  xitk_window_define_window_cursor (setup->nw.xwin, xitk_cursor_default);
  xitk_slider_set_pos (setup->w[_S_slider], max - setup->first_displayed);
}

static void setup_apply (xitk_widget_t *w, void *data, int state, unsigned int modifier) {
  xui_setup_t *setup = data;

  (void)state;
  if ((setup->num_wg > 0) && (w != setup->w[_S_close])) {
    int need_restart = 0;
    _widget_triplet_t *wt = setup->wg, *end = wt + setup->num_wg;

    for (; wt < end; wt++) {
      if (!wt->changed)
        continue;
      wt->changed = 0;
      if (!need_restart && !wt->cfg.callback_data && !wt->cfg.callback)
        need_restart = 1;
      if (wt->cfg.type == XINE_CONFIG_TYPE_STRING) {
        xine_cfg_entry_t entry;
        /**          v-- will not be written to. */
        wt->cfg.str_value = (char *)xitk_inputtext_get_text (wt->w[_T_widget]);
        xine_config_update_entry (setup->nw.gui->xine, &wt->cfg);
        wt->cfg.str_value = xine_config_lookup_entry (setup->nw.gui->xine, wt->cfg.key, &entry) ? entry.str_value : (char *)"";
      } else {
        xine_config_update_entry (setup->nw.gui->xine, &wt->cfg);
      }
    }
    xine_config_save (setup->nw.gui->xine, setup->nw.gui->cfg_file);

    if (w != setup->w[_S_ok]) /* reöoad/redisplay. */
      setup_change_section (setup->w[_S_tabs], setup, setup->sections.shown, modifier);

    if (need_restart)
      setup->dialog = xitk_window_dialog_3 (setup->nw.gui->xitk,
        setup->nw.xwin,
        gui_layer_above (setup->nw.gui, NULL), 400, _("Important Notice"), NULL, NULL,
        XITK_LABEL_OK, NULL, NULL, NULL, 0, ALIGN_CENTER,
        "%s", _("You changed some configuration value which require to restart xine to take effect."));
  }
  if (w != setup->w[_S_apply])
    setup->exit (setup);
}

/*
 * collect config categories, setup tab widget
 */
static int _section_cmp (void *a, void *b) {
  const _setup_string4_t * const *d = (const _setup_string4_t * const *)a;
  const _setup_string4_t * const *e = (const _setup_string4_t * const *)b;
  const _setup_string4_t *n1 = *d, *n2 = *e;
  const union { uint8_t b[4]; uint32_t w; } end = { .b = { [3] = 255 }};
  uint32_t u;
  /* we just test for duplicates, so endian does not matter. */
  do {
    if (n1->v < n2->v)
      return -1;
    if (n1->v > n2->v)
      return 1;
    u = n1->v & n2->v;
    n1++;
    n2++;
  } while (u & end.w);
  return 0;
}

#if defined(XINE_SARRAY) && (XINE_SARRAY >= 3)
static unsigned int _section_hash (void *a) {
  const _setup_string4_t * const *d = (const _setup_string4_t * const *)a;
  const _setup_string4_t *n1 = *d;
  const union { uint8_t b[4]; uint32_t w; } end = { .b = { [3] = 255 }};
  uint32_t v = 0, u;

  do {
    v ^= n1->v;
    u = n1->v;
    n1++;
  } while (u & end.w);
  v += v >> 20;
  v += v >> 10;
  v ^= v >> 5;
  return v & 31;
}
#endif

static void setup_sections (xui_setup_t *setup) {
  xine_cfg_entry_t    entry;
  int                 res;
  _setup_string4_t   *q, *e;
  xine_sarray_t      *a = xine_sarray_new (SETUP_MAX_SECTIONS, _section_cmp);

#ifdef XINE_SARRAY_MODE_UNIQUE
  xine_sarray_set_mode (a, XINE_SARRAY_MODE_UNIQUE);
#endif
#if defined(XINE_SARRAY) && (XINE_SARRAY >= 3)
  xine_sarray_set_hash (a, _section_hash, 32);
#endif

  q = setup->namebuf;
  e = q + sizeof (setup->namebuf) / sizeof (setup->namebuf[0]);
  setup->sections.have = 0;

  for (res = xine_config_get_first_entry (setup->nw.gui->xine, &entry);
       res;
       res = xine_config_get_next_entry (setup->nw.gui->xine, &entry)) {
    uint32_t nlen;
    int i;

    /* assume that entries without a human readable description
     * are for internal use only, like our own window position reminders. */
    if (!entry.description)
      continue;
    /* skip entries that are (yet) not in use. */
    if (entry.type == XINE_CONFIG_TYPE_UNKNOWN)
      continue;
    /* get section name, and want it to be non empty. */
    if (!entry.key[nlen = xitk_find_0_or_byte (entry.key, '.')])
      continue;
    /* paranoia. */
    if ((nlen >> 2) + 1 > (uint32_t)(e - q))
      continue;
    setup->sections.nlen[setup->sections.have] = nlen;
    setup->sections.name[setup->sections.have] = q->s;
    q[nlen >> 2].v = 0;
    memcpy (q->s, entry.key, nlen);
    /* we already have this one? */
#ifdef XINE_SARRAY_MODE_UNIQUE
    i = xine_sarray_add (a, setup->sections.name + setup->sections.have);
#else
    /* rare old libxine */
    i = ~xine_sarray_binary_search (a, setup->sections.name + setup->sections.have);
    if (i >= 0)
      i = xine_sarray_add (a, setup->sections.name + setup->sections.have);
#endif
    if (i < 0) {
      const char * const *s4 = xine_sarray_get (a, ~i);
      setup->sections.num[s4 - setup->sections.name] += 1;
    }
    /* no, add it. */
    else if (setup->sections.have < SETUP_MAX_SECTIONS) {
      /* Aargh. xitk_noskin_tabs_create () does not copy the strings.
       * Give it a "permanent" copy without the trailing . ("foo"). */
      q += (nlen >> 2) + 1;
      setup->sections.num[setup->sections.have] = 1;
      setup->sections.have++;
    }
  }

#if defined(XINE_SARRAY) && (XINE_SARRAY >= 4)
  if (setup->nw.gui->verbosity >= 2) {
    unsigned int q = xine_sarray_hash_quality (a);
    printf ("gui.setup.sections.hash_quality (%u.%u%%).\n", q / 10u, q % 10u);
  }
#endif
  xine_sarray_delete (a);

  {
    int max = 1, i;

    for (i = 0; i < setup->sections.have; i++) {
      setup->sections.first_shown[i] = 0;
      if (max < setup->sections.num[i])
        max = setup->sections.num[i];
    }
    setup->max_wg = (max + 7) & ~7;
    free (setup->wg);
    setup->wg = malloc (setup->max_wg * sizeof (*setup->wg));
  }

  {
    xitk_tabs_widget_t tab = {
      .nw = {
        .wl = setup->nw.wl,
        .userdata = setup,
        .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE
      },
      .num_entries = setup->sections.have,
      .entries     = setup->sections.name,
      .callback    = setup_change_section
    };

    setup->w[_S_tabs] = xitk_noskin_tabs_create (&tab, 15, 24, WINDOW_WIDTH - 30, tabsfontname);
  }
  setup->th = xitk_get_widget_height (setup->w[_S_tabs]) - 1;

  setup->num_wg = 0;
  setup->first_displayed = 0;
}

/*
 *
 */
static void setup_nextprev_wg(xitk_widget_t *w, void *data, int pos, unsigned int modifier) {
  xui_setup_t *setup = data;
  int max;

  (void)w;
  (void)modifier;
  max = setup->num_wg - MAX_DISPLAY_WIDGETS;
  if (max < 0)
    max = 0;
  setup_paint_widgets (setup, max - pos);
}


/*
 * Create setup panel window
 */
void setup_main (xitk_widget_t *mode, void *data) {
  gGui_t *gui = data;
  xui_setup_t *setup;
  xitk_font_t *fs;

  if (!gui)
    return;

  setup = gui->setup;
  if (mode == XUI_W_OFF) {
    if (!setup)
      return;
    setup->exit (setup);
    return;
  } else if (mode == XUI_W_ON) {
    if (setup) {
      gui_raise_window (gui, setup->nw.xwin);
      return;
    }
  } else { /* toggle */
    if (setup) {
      setup->exit (setup);
      return;
    }
  }

  setup = calloc (1, sizeof (*setup));
  if (!setup)
    return;

  setup->nw.gui = gui;
  setup->exit = _setup_exit;

  if (xitk_init_NULL ()) {
    setup->nw.skin = NULL;
    setup->nw.wfskin = NULL;
    setup->nw.adjust = NULL;
  }

  /* Create window */
  setup->nw.title = _("xine Setup");
  setup->nw.id = "setup";
  setup->nw.wr.x = 80;
  setup->nw.wr.y = 80;
  setup->nw.wr.width = WINDOW_WIDTH;
  setup->nw.wr.height = WINDOW_HEIGHT;
  if (gui_window_new (&setup->nw) < 0) {
    free (setup);
    return;
  }

  fs = xitk_font_load_font (setup->nw.gui->xitk, fontname);
  setup->fh = xitk_font_text_height (fs, " ÜJg", 4) + 4;

  setup_sections (setup);

  setup->main.x = 15;
  setup->main.y = 24 + setup->th;
  setup->main.width = WINDOW_WIDTH - 2 * 15;
  setup->main.height = MAX_DISPLAY_WIDGETS * (FRAME_HEIGHT + 3) + 2 * 15;

  setup->sl.width = 16;
  setup->sl.height = MAX_DISPLAY_WIDGETS * (FRAME_HEIGHT + 3);
  setup->sl.x = setup->main.x + setup->main.width - setup->sl.width - 5;
  setup->sl.y = setup->main.y + 15;

  setup->frame.x = setup->main.x + 18;
  setup->frame.y = setup->main.y + 15;
  setup->frame.width = setup->main.width - setup->sl.width - 5 - 2 * 18;
  setup->frame.height = FRAME_HEIGHT;
  xitk_image_draw_rectangular_box (setup->nw.bg,
    setup->main.x, setup->main.y, setup->main.width, setup->main.height, XITK_DRAW_OUTTER);
  xitk_window_set_background_image (setup->nw.xwin, setup->nw.bg);

  {
    xitk_slider_widget_t sl = {
      .nw = { .wl = setup->nw.wl, .userdata = setup, .add_state = XITK_WIDGET_STATE_CLEAR },
      .min             = 0,
      .max             = 1,
      .step            = 1,
      .type            = XITK_HVSLIDER,
      .motion_callback = setup_nextprev_wg
    };

    setup->w[_S_slider] =  xitk_noskin_slider_create (&sl, setup->sl.x, setup->sl.y, setup->sl.width, setup->sl.height);
  }

  setup_section_widgets (setup, 0);
  setup_paint_widgets (setup, 0);

  {
    const char *label = _("(*)  you need to restart xine for this setting to take effect");
    int len = xitk_font_text_width (fs, label, -1);
    xitk_label_widget_t lbl = {
      .nw = { .wl = setup->nw.wl, .userdata = setup, .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE },
      .label = label
    };

    xitk_noskin_label_create (&lbl,
      (WINDOW_WIDTH - len) >> 1, setup->main.y + setup->main.height, len + 3, 18, fontname);
  }

  xitk_font_unload_font(fs);

  {
    uint32_t style = XITK_DRAW_SAT (setup->nw.gui->gfx_saturation);
    char buf[256];
    xitk_labelbutton_widget_t lb = {
      .nw = { .wl = setup->nw.wl, .userdata = setup, .add_state = XITK_WIDGET_STATE_VISIBLE, .tips = buf },
      .button_type = CLICK_BUTTON,
      .align       = ALIGN_CENTER,
      .label    = _("Apply"),
      .callback = setup_apply,
      .style    = XITK_DRAW_B | style
    };

    snprintf (buf, sizeof (buf), "%s-S", xitk_gettext ("Ctrl"));
    setup->w[_S_apply] = xitk_noskin_labelbutton_create (&lb,
      (WINDOW_WIDTH - 100) >> 1, WINDOW_HEIGHT - (23 + 15), 100, 23,
      XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_INV, tabsfontname);

    lb.nw.add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE;

    lb.label    = _("Close");
    lb.style    = XITK_DRAW_R | style;
    snprintf (buf, sizeof (buf), "Esc; %s", _("Discard changes and dismiss the window."));
    setup->w[_S_close] = xitk_noskin_labelbutton_create (&lb,
      WINDOW_WIDTH - (100 + 15), WINDOW_HEIGHT - (23 + 15), 100, 23,
      XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_INV, tabsfontname);

    lb.label    = _("OK");
    lb.style    = XITK_DRAW_G | style;
    lb.nw.tips  = _("Apply the changes and close the window.");
    setup->w[_S_ok] = xitk_noskin_labelbutton_create (&lb,
      15, WINDOW_HEIGHT - (23 + 15), 100, 23,
      XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_INV, tabsfontname);
  }
  setup->nw.key = xitk_be_register_event_handler ("setup", setup->nw.xwin, setup_event, setup, NULL, NULL);

  gui_raise_window (setup->nw.gui, setup->nw.xwin);

  setup->nw.gui->setup = setup;
}
