/* prefixdbentry.cc
 * This file belongs to Worker, a file manager for UN*X/X11.
 * Copyright (C) 2010 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "prefixdbentry.hh"
#include <algorithm>
#include <iostream>
#include <fstream>
#include <sstream>
#include "datei.h"
#include <aguix/util.h>
#include "configtokens.h"
#include "configheader.h"
#include "configparser.hh"

PrefixDBEntry::PrefixDBEntry( const std::string &prefix ) : m_prefix( prefix )
{
}

PrefixDBEntry::~PrefixDBEntry()
{
}

bool PrefixDBEntry::operator<( const PrefixDBEntry &rhs ) const
{
    return m_prefix < rhs.m_prefix;
}

void PrefixDBEntry::storeAccess( const std::string &value )
{
    std::list< std::pair< std::string, float > >::iterator it1;

    for ( it1 = m_strings.begin();
	  it1 != m_strings.end();
	  it1++ ) {
	if ( it1->first == value ) break;
    }
    
    if ( it1 != m_strings.end() ) {
	it1->second++;
    } else {
	m_strings.push_back( std::make_pair( value, 1.0 ) );
    }
}

std::string PrefixDBEntry::getBestHit() const
{
    float t1;
    return getBestHit( t1 );
}

std::string PrefixDBEntry::getBestHit( float &count ) const
{
    std::list< std::pair< std::string, float > >::const_iterator it1, best_hit;

    if ( m_strings.empty() ) return "";

    best_hit = m_strings.begin();

    for ( it1 = m_strings.begin();
	  it1 != m_strings.end();
	  it1++ ) {
	if ( it1->second > best_hit->second ) {
	    best_hit = it1;
	}
    }

    count = best_hit->second;
    return best_hit->first;;
}

const std::string &PrefixDBEntry::getPrefix() const
{
    return m_prefix;
}

const std::list< std::pair< std::string, float > > &PrefixDBEntry::getStrings() const
{
    return m_strings;
}

void PrefixDBEntry::age( float factor )
{
    std::list< std::pair< std::string, float > >::iterator it1;

    for ( it1 = m_strings.begin();
	  it1 != m_strings.end();
	  it1++ ) {
	it1->second *= factor;
    }
}

void PrefixDBEntry::write( std::ofstream &ofile ) const
{
    ofile << "prefix=" << m_prefix << std::endl;

    std::list< std::pair< std::string, float > >::const_iterator it1;

    for ( it1 = m_strings.begin();
	  it1 != m_strings.end();
	  it1++ ) {
	ofile << "value=" << it1->second << ":" << it1->first << std::endl;
    }
}

void PrefixDBEntry::write( Datei &fh ) const
{
    fh.configPutPairString( "prefix", m_prefix.c_str() );

    std::list< std::pair< std::string, float > >::const_iterator it1;

    for ( it1 = m_strings.begin();
          it1 != m_strings.end();
          it1++ ) {
        std::string f1 = AGUIXUtils::formatStringToString( "%f", it1->second );
        fh.configPutPairString( "value", f1.c_str() );
        fh.configPutPairString( "name", it1->first.c_str() );
    }
}

bool PrefixDBEntry::read( std::ifstream &ifile )
{
    bool failed = true;
    std::string line;

    m_strings.clear();
    m_prefix = "";

    if ( std::getline( ifile, line ) ) {
	if ( line.compare( 0, std::string( "prefix=" ).length(), "prefix=" ) == 0 ) {
	    m_prefix = std::string( line, std::string( "prefix=" ).length() );
	    failed = false;
	}
    }

    if ( ! failed ) {
	while ( ! failed && std::getline( ifile, line ) ) {
	    if ( line == "end_entry" ) break;

	    if ( line.compare( 0, std::string( "value=" ).length(), "value=" ) == 0 ) {
		std::string value( line, std::string( "value=" ).length() );
		std::string::size_type pos = value.find( ':' );
		if ( pos != std::string::npos ) {
		    std::string count( value, 0, pos );
		    std::string str1( value, pos + 1 );
		    
		    m_strings.push_back( std::make_pair( str1, atof( count.c_str() ) ) );
		} else {
		    failed = true;
		}
	    } else {
		failed = true;
	    }
	}
    }

    return failed;
}

bool PrefixDBEntry::read( FILE *fp )
{
    if ( fp == NULL ) return true;
    std::string line;

    m_strings.clear();
    m_prefix = "";

    int found_error = 0;
    std::string str1;
    float value = -1.0;

    for (;;) {
        if ( worker_token == PREFIX_WCP ) {
            readtoken();

            if ( worker_token != '=' ) {
                found_error = 1;
                break;
            }
            readtoken();

            if ( worker_token != STRING_WCP ) {
                found_error = 1;
                break;
            }

            str1 = yylval.strptr;
            m_prefix = str1;

            readtoken();

            if ( worker_token != ';' ) {
                found_error = 1;
                break;
            }
            readtoken();
        } else if ( worker_token == VALUE_WCP ) {
            readtoken();

            if ( worker_token != '=' ) {
                found_error = 1;
                break;
            }
            readtoken();
            if ( worker_token != STRING_WCP ) {
                found_error = 1;
                break;
            }

            str1 = yylval.strptr;
            if ( sscanf( yylval.strptr, "%f", &value ) == 1 ) {
            } else {
                value = -1.0;
            }
            readtoken();

            if ( worker_token != ';' ) {
                found_error = 1;
                break;
            }
            readtoken();
        } else if ( worker_token == NAME_WCP ) {
            readtoken();

            if ( worker_token != '=' ) {
                found_error = 1;
                break;
            }
            readtoken();

            if ( worker_token != STRING_WCP ) {
                found_error = 1;
                break;
            }

            str1 = yylval.strptr;
            readtoken();

            if ( worker_token != ';' ) {
                found_error = 1;
                break;
            }
            readtoken();

            if ( value > 0.0 ) {
                m_strings.push_back( std::make_pair( str1, value ) );
            }
        } else {
            break;
        }
    }

    return found_error > 0;
}
