/*
*  RAL -- Rubrica Addressbook Library
*  file: load.c
*  
*  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <stdlib.h>
#include <time.h>
#include <libintl.h>
#include <glib.h>
#include <libxml/tree.h>

#include "load.h"
#include "io.h"
#include "../../card.h"
#include "../../personal.h"
#include "../../company.h"
#include "../../contact.h"
#include "../../address.h"
#include "../../group.h"
#include "../../net.h"
#include "../../notes.h"
#include "../../telephone.h"
#include "../../error.h"


/*
  RubricaDoc file structure:
     version:    rubrica's version that made the addressbook file
     fileformat: #4
     doctype:    addressbook

  <?xml version="1.0" encoding="UTF-8"?>
  <Rubrica version="" fileformat="" doctype="">        root
     <Card                                             root child 
            name=""                                    the card name
            locked=""                               is a locked card?
            deleted=""                                 was card deleted?
            rank=""                                    card rank (1..5)
            id="pcxxx"                                 card id   (es:  pc643)
	    created=""                                 creation date
            last_change=""                             last change date
	    type="personal">                           card's type 
         <Groups>                                      card child 
             <Group owner="rubrica" pixmap="">...</Group>   groups children
	     <Group owner="user" pixmap="">......</Group>
	     ...
         </Groups>       
	 <Refs>
	     <Ref refid="card_id" refinfo="references desc">.........</Ref>
	     ...
         </Refs> 
	 <Addresses>                                   card child
             <Address type="">                         addresses children
                 <Street number="">....</Street>       address child
	         <City zip="">.........</City>
	         <Province>............</Province>
	         <State>...............</State>
	         <Country>.............</Country>
 	      </Address>
	     ...
         </Addresses>
	 <Net>
	     <Uri type="web">..........</Uri>
	     <Uri type="email">........</Uri>
	     <Uri type="aim">..........</Uri>
	     <Uri type="msn">..........</Uri>
	     <Uri type="yahoo">........</Uri>
	     <Uri type="icq">..........</Uri>
	     <Uri type="jabber">.......</Uri>	     
	     ...
	 </Net>
	 <TelephoneNumbers>
	     <Telephone type="home">...</Telephone>
	     <Telephone type="work">...</Telephone>
	     <Telephone type="fax">....</Telephone>
	     ...
 	 </TelephoneNumbers>
         <Data>                                        card child
             <FirstName>...............</FirstName>    data child 
	     <MiddleName>..............</MiddleName>     "    "
	     <LastName>................</LastName>                    
	     <Nickname>................</NickName>                    
	     <Profession>..............</Profession>                  
 	     <NamePrefix>..............</NamePrefix>                  
	     <Title>...................</Title>                       
	     <Photo type="">...........</Photo>
	     <Birthday known="" day="" month="" year=""/>  
         </Data>
	 <Work>
	     <Assignment>..............</Assignment>
	     <Organization>............</Organization>
             <Department>..............</Department>
	     <SubDepartment>...........</SubDepartment>
	     <ManagerName>.............</ManagerName>
	     <ManagerPhone>............</ManagerPhone>
	     <CollabratorName>.........</CollabratorName>
	     <CollaboratorPhone>.......</CollabratorPhone>
         </Work>
	 <Notes married="">
             <SpouseName>..............</SpouseName>
	     <Child>...................</Child>
	     <OtherNotes>..............</OtherNotes>
	     <PublicKey>...............</PublicKey>
	     <SpouseBirthday known="" day="" month="" year=""/>
	     <Anniversary known="" day="" month="" year""/>
         </Notes>
     </Card>

     <Card                                             root child 
            name=""                                    the card name
            locked=""                               is a locked card?
            deleted=""                                 was card deleted?
            rank=""                                    card rank (1..5)
            id="ccxxx"                                 card id   (es: cc1039)
	    created=""                                 creation date
            last_change=""                             last change date
	    type="company">                            card's type 
         <Groups>                                      card child 
             <Group>...................</Group>        groups children
	     ...
         </Groups>       
	 <Refs>
	     <Ref refid="card_id" refinfo="references desc">.........</Ref>
	     ...
         </Refs> 
         <Company>
             <CompanyName>.............</CompanyName>
	     <Notes>...................</Notes>
         </Company>
	 <Addresses>                                   card child
             <Address type="">                         addresses children
                 <Street number="">....</Street>       address child
	         <City zip="">.........</City>
	         <Province>............</Province>
	         <State>...............</State>
	         <Country>.............</Country>
 	      </Address>
	     ...
         </Addresses>
	 <Net>
	     <Uri type="web">..........</Uri>
	     <Uri type="email">........</Uri> 
	     ...
	 </Net>
	 <TelephoneNumbers>
	     <Telephone type="green">......</Telephone>
	     <Telephone type="operator">...</Telephone>
	     <Telephone type="fax">........</Telephone>
	     ...
 	 </TelephoneNumbers>
	 <CompanyContacts>
	     <Contact ref_id="id_personal">first name, last name</Contact>
	     ...
	 <CompanyContacts>
     </Card>
     ...
  </Rubrica>


  Rubrica
     |
     | children
     |
     v   next         next         next 
   Card-------->Card-------->....------->Card
     |
     | children
     |
     v   next                
   Groups--->Data--->Addresses--->Net--->....Phone-->Work-->Notes
     |        \          |         |           |        |      |      
     | child  | child    | child   |           |        |      |     
     |        |          v         v           v        v      v     
     |        |         ...       ...         ...      ...     ...    ...   
     v        v
   Group    FirstName--->MiddleName--->...

   pcxxx => personal card #xxx
   ccxxx => company  card #xxx
*/

void   r_real_load_card (RCard* card, xmlNodePtr xmlcard);
gchar *r_get_card_name  (xmlNodePtr node, RError *err);


void r_read_web     (RCard *card, xmlNodePtr node);
void r_read_email   (RCard *card, xmlNodePtr node);
void r_read_address (RCard *card, xmlNodePtr node);


gboolean reading_old_card = FALSE;



gchar* 
r_get_card_name(xmlNodePtr node, RError *err)
{
  gchar* ret = NULL;

  ret = r_io_get_prop(node, "name", err);

  if (!ret)
    {
//      r_log_write(err);

      return NULL;
    }

  return (gchar*) ret;
}



void 
r_read_infos(RCard *card, xmlNodePtr node)
{
  xmlNodePtr xmlcard = NULL;
  RError err;
  gchar* id;
  time_t created, changed;

  g_return_if_fail(IS_R_CARD(card));

  xmlcard = r_io_get_node(node, (xmlChar*) "Card");

  if (xmlcard)
    {
      gchar *name, *tmp;
      gboolean locked, deleted; // , marked;
      RRate rate;
 
      name    = r_get_card_name(xmlcard, &err);
      locked  = r_io_get_bool(xmlcard, "locked", &err);
      deleted = r_io_get_bool(xmlcard, "deleted", &err);
      tmp     = r_io_get_prop(xmlcard, "rate", &err); 

      if (tmp)
	{
	  rate = (RRate) atoi(tmp);
	  g_free(tmp);
	}
      else 
	rate = R_RATE_NORMAL;

      id = r_io_get_prop(xmlcard, "id", &err);
      if (id && (g_ascii_strcasecmp(id, "0") != 0))
	{
	  gchar* type = NULL;

	  type = r_io_get_prop(xmlcard, "type", &err);
	  
	  r_card_reassign_id(card, atol(id));
	  g_free(id);
	}
      else // I'm readind an old card, no id is needed to be set
	{
	  locked  = r_io_get_bool(xmlcard, "deleting", &err);
	  deleted = FALSE;
	}
   
      g_object_set(card, 
		   "card-name", name, "card-locked", locked, 
		   "card-deleted", deleted, "card-rate", rate, NULL);
      
      created = r_io_get_date(xmlcard, "created", &err);
      changed = r_io_get_date(xmlcard, "last_change", &err);
	    
      g_object_set(card, "card-created", created, "card-changed", changed, 
		   NULL);
    }
  else
    {
      /* TODO 
	 
      segnalare l'errore, far scegliere all'utente se 
      continuare a caricare automaticamente il file 
      o avviare un caricamento assistito
      */	 
    }
}



void 
r_read_groups(RCard *card, xmlNodePtr cardnode)
{
  xmlNodePtr xmltablegroups = NULL;
  xmlNodePtr xmlgroup;
  RError err;

  RGroup* group;
  gchar* name;
  gchar* owner;
  gchar* pixmap;
  
  /*  
      <Groups>                                          card's child 
          <Group owner="" pixmap="">........</Group>    groups children
          ...
      </Groups>       
  */

  g_return_if_fail(IS_R_CARD(card));

  xmltablegroups = r_io_get_node(cardnode, (xmlChar*) "Groups");
  
  if (xmltablegroups)
    {
      xmlgroup = xmltablegroups->children;
      if (xmlIsBlankNode(xmlgroup))
	xmlgroup = xmlgroup->next;
      
      while (xmlgroup)
	{	  
	  if (xmlIsBlankNode(xmlgroup))
	    xmlgroup = xmlgroup->next;		  
	  
	  group = r_group_new();
	  
	  name   = r_io_get_content(xmlgroup, &err);
	  owner  = r_io_get_prop(xmlgroup, "owner", &err);
	  pixmap = r_io_get_prop(xmlgroup, "pixmap", &err);
		  
	  g_object_set(group, "group-name", name, "group-label", name,
		       "group-owner", owner, "group-pixmap", 
		       pixmap, "enabled", TRUE, NULL); 
		       
	  r_card_add_group(card, group);
	  
	  if (name)   g_free(name);
	  if (owner)  g_free(owner);
	  if (pixmap) g_free(pixmap);

	  xmlgroup = xmlgroup->next;
	  if (xmlIsBlankNode(xmlgroup))
	    xmlgroup = xmlgroup->next;
	} 
    }
  else // may be i'm reading an old card
    {
      name = r_io_get_prop(cardnode, "group", &err);
      
      if (name)
	{
	  group = r_group_new();
	  g_object_set(group, "group-name", name, "group-owner", "rubrica",  
		       "group-pixmap", "", "enabled", TRUE, NULL); 
		       
	  r_card_add_group(card, group);	  
	}
    }
}


void 
r_read_refs (RCard *card, xmlNodePtr cardnode)
{
  xmlNodePtr xmlrefs = NULL;
  xmlNodePtr xmlref;
  RError err;

  /* 
     <Refs>                                                    card's child
         <Ref refid="card_id">references's description</Ref>   refs children
	 ....
     </Refs>      
  */

  g_return_if_fail(IS_R_CARD(card));

  xmlrefs = r_io_get_node(cardnode, (xmlChar*) "Refs");
  
  if (xmlrefs)
    {
      xmlref = xmlrefs->children;
      if (xmlIsBlankNode(xmlref))
	xmlref = xmlref->next;
      
      while (xmlref)
	{	  
	  RRef* ref;
	  gchar* ref_id, *desc;
	  glong id = 0L;

	  if (xmlIsBlankNode(xmlref))
	    xmlref = xmlref->next;		  
	  	  
	  desc   = r_io_get_content(xmlref, &err);
	  ref_id = r_io_get_prop(xmlref, "refid", &err);
	  
	  if (ref_id)
	    id = atol(ref_id);
	    
	  ref = r_ref_new(id);
	  g_object_set(ref, "ref-info", desc, NULL); 
		       
	  r_card_add_ref(card, ref);
	  
	  if (ref_id) g_free(ref_id);
	  if (desc)   g_free(desc);

	  xmlref = xmlref->next;
	  if (xmlIsBlankNode(xmlref))
	    xmlref = xmlref->next;
	} 
    }
}


void 
r_read_contact(RPersonalCard *card, xmlNodePtr cardnode)
{
  RError err;
  RContact* contact;
  xmlNodePtr xmldata = NULL;

  g_return_if_fail(IS_R_PERSONAL_CARD(card));

  contact = r_contact_new();
  if (!IS_R_CONTACT(contact))
    {
      r_personal_card_set_contact(card, NULL);
      return;
    }

  xmldata = r_io_get_node(cardnode, (xmlChar*) "Data");
  
  if (xmldata)  
    {   
      gchar *day, *month, *year;
      gchar *first, *middle, *last, *nick, *genre;
      gchar *prefix, *prof, *title, *photo;
      
      first  = r_io_get(xmldata, "FirstName", &err);
      middle = r_io_get(xmldata, "MiddleName", &err);
      last   = r_io_get(xmldata, "LastName", &err);
      nick   = r_io_get(xmldata, "NickName", &err);
      prof   = r_io_get(xmldata, "Profession", &err);
      prefix = r_io_get(xmldata, "NamePrefix", &err);
      title  = r_io_get(xmldata, "Title", &err);
      genre  = r_io_get(xmldata, "Genre", &err);
      photo  = r_io_get(xmldata, "Photo", &err);

      g_object_set(contact, 
		   "first-name", first, "middle-name", middle, 
		   "last-name", last, "nick-name", nick, 
		   "prefix", prefix, "profession", prof, 
		   "genre", genre, "title", title, "photo", photo, NULL);

      g_free(first);
      g_free(middle);
      g_free(last);
      g_free(nick);
      g_free(prefix);
      g_free(prof);
      g_free(title);
      g_free(genre);
      g_free(photo);

      r_io_get_calendar_from (xmldata, "Birthday", 
			      &day, &month, &year, &err);
      if ((g_ascii_strcasecmp(day,   "BadDay") == 0)   &&  
	  (g_ascii_strcasecmp(month, "BadMonth") == 0) && 
	  (g_ascii_strcasecmp(year,  "BadYear") == 0)) // reading an old node ?
	{
	  xmlNodePtr node;
	  time_t birth;
	  struct tm time;

	  node = r_io_get_node(xmldata, (xmlChar*) "FirstName");
	  if (node)
	    {
	      if(xmlHasProp(node, (xmlChar*) "know_birthday"))
		{
		  if (r_io_get_bool(node, "know_birthday", &err))
		    {
		      birth = r_io_get_date(node, "birthday", &err);
		      
		      localtime_r(&birth, &time);
		      
		      r_contact_set_birthday (contact, time.tm_mday,
					      time.tm_mon, time.tm_year);
		    }
		}
	    }
	}
      else
	r_contact_set_birthday (contact, atoi(day), atoi(month),
				      atoi(year));

      r_personal_card_set_contact(card, contact);
    }
}


void 
r_read_work(RPersonalCard *card, xmlNodePtr cardnode)
{
  RError err;
  xmlNodePtr xmlwork = NULL;
  xmlNodePtr old = NULL;

  g_return_if_fail(IS_R_PERSONAL_CARD(card));

  xmlwork = r_io_get_node(cardnode, (xmlChar*) "Work");

  if (xmlwork)
    {     
      RWork* work;
      gchar *assignment = NULL, *org = NULL, *dep = NULL, *subdep = NULL;
      gchar *manager = NULL, *mphone = NULL;
      gchar *collaborator =NULL, *cphone = NULL;
      
      assignment   = r_io_get(xmlwork, "Assignment", &err);      
      org          = r_io_get(xmlwork, "Organization", &err);
      dep          = r_io_get(xmlwork, "Department", &err);
      subdep       = r_io_get(xmlwork, "SubDepartment", &err);
      manager      = r_io_get(xmlwork, "ManagerName", &err);
      mphone       = r_io_get(xmlwork, "ManagerPhone", &err);
      collaborator = r_io_get(xmlwork, "CollaboratorName", &err);
      cphone       = r_io_get(xmlwork, "CollaboratorPhone", &err);

      old = r_io_get_node(xmlwork, (xmlChar*) "SecretaryName");
      if (old) // reading old card
	{
	  collaborator = r_io_get(xmlwork, "SecretaryName", &err);
	  cphone       = r_io_get(xmlwork, "SecretaryPhone", &err);
	}

      if (assignment || org || dep || subdep || manager || 
	  mphone || collaborator || cphone)
	{
	  work = r_work_new();
	  
	  if (!IS_R_WORK(work))
	    return;

	  g_object_set(work, 
		       "assignment", assignment, "organization", org, 
		       "department", dep, "sub-department", subdep, 
		       "manager-name", manager, "manager-phone", mphone, 
		       "collaborator", collaborator, 
		       "collaborator-phone", cphone,
		       NULL);
	  
	  g_free(assignment);
	  g_free(org);
	  g_free(dep);
	  g_free(subdep);
	  g_free(manager);
	  g_free(mphone);
	  g_free(collaborator);
	  g_free(cphone);
	   
	  r_personal_card_set_work(card, work);
	}	
    }
}



void  
r_read_addresses(RCard *card, xmlNodePtr cardnode)
{
  RAddress* address = NULL;
  RError err;
  xmlNodePtr xmladd = NULL;
  xmlNodePtr xmladdress;
  gchar *tmp;

  g_return_if_fail(IS_R_CARD(card));

  xmladd = r_io_get_node(cardnode, (xmlChar*) "Addresses");

  if (xmladd)
    {
      RAddressType type;
      gchar *street = NULL, *number = NULL, *city = NULL, *zip = NULL;
      gchar *province = NULL, *state = NULL, *country = NULL;

      /* child is the head of an <address> list */
      xmladdress = xmladd->children;  
      if (xmlIsBlankNode(xmladdress))
	xmladdress = xmladdress->next;
      
      while(xmladdress)
	{	  
	  /* The address's data:

	    <Address type="">                         addresses children
                <Street number="">....</Street>       address child 
		...
	    </Address>
	  */

	  /* get address type --- <Address type="">  */
	  tmp      = r_io_get_prop(xmladdress, "type", &err);
	  type     = r_address_lookup_str2enum(tmp);
	  street   = r_io_get(xmladdress, "Street", &err);
	  number   = r_io_get_prop_from(xmladdress, "Street", "number",
					&err);
	  city     = r_io_get(xmladdress, "City", &err);
	  zip      = r_io_get_prop_from(xmladdress, "City", "zip", &err);
	  province = r_io_get(xmladdress, "Province", &err);
	  state    = r_io_get(xmladdress, "State", &err);
	  country  = r_io_get(xmladdress, "Country", &err);
	  
	  if (street || number || city || zip || province || state || country)
	    {
	      address = r_address_new();
	      
	      if (!IS_R_ADDRESS(address))
		{
		  g_warning("address obj get wrong type");
		  
		  return;
		}	      
	      
	      g_object_set(address, "address-type", type, "street", street,  
			   "street-number", number, "city", city, "zip", zip,  
			   "province", province, "state", state,  
			   "country", country, NULL); 
	      
	      r_card_add_address(card, address);
	    }
	  
	  xmladdress = xmladdress->next;
	  if (xmlIsBlankNode(xmladdress)) 
	    xmladdress = xmladdress->next;
	}
    }
}


void
r_read_net(RCard *card, xmlNodePtr cardnode)
{
  RNetAddress* net;
  RError err;
  xmlNodePtr xmlnet_child;
  xmlNodePtr xmlnet = NULL;
  gchar* net_data;
  gchar* net_type;

  /*
    <Net>
       <Uri type="web">..........</Uri>
       <Uri type="email">........</Uri> 
       <Uri type="icq">..........</Uri>
       ...
    </Net>
  */
  g_return_if_fail(IS_R_CARD(card));

  xmlnet = r_io_get_node(cardnode, (xmlChar*) "Net");

  if (xmlnet)
    {
      xmlnet_child = xmlnet->children;
      if (xmlIsBlankNode(xmlnet_child))
	xmlnet_child = xmlnet_child->next;

      while (xmlnet_child)
	{
	  RNetAddressType type;

	  if (xmlIsBlankNode(xmlnet_child))
	    xmlnet_child = xmlnet_child->next;
	  
	  net_data = r_io_get_content(xmlnet_child, &err);
	  net_type = r_io_get_prop(xmlnet_child, "type", &err);

	  if (net_data)
	    {
	      net = r_net_address_new();
	      
	      if(!IS_R_NET_ADDRESS(net))
		{
		  g_warning("net obj get wrong type");
		  
		  g_free(net_type);
		  g_free(net_data);

		  return;
		}
	      
	      type = r_net_address_encode_type(net_type);
	      g_object_set(net, "url", net_data, "url-type", type, NULL);
	      r_card_add_net_address(card, net);
	      
	      g_free(net_data);
	      g_free(net_type);
	    }	  

	  xmlnet_child = xmlnet_child->next;
	  if (xmlIsBlankNode(xmlnet_child)) 
	    xmlnet_child = xmlnet_child->next;
	}
    }
}

 

void  
r_read_web(RCard *card, xmlNodePtr node) 
{ 
  RError err; 
  xmlNodePtr child; 
  RNetAddress* net;
  xmlNodePtr xmlweb = NULL; 
  gchar *tmp; 
  
  xmlweb = r_io_get_node(node, (xmlChar*) "WebAddresses"); 

  if (xmlweb) 
    {
      child = xmlweb->children; 
      if (xmlIsBlankNode(child)) 
	child = child->next; 

      while (child)
	{
	  if (xmlIsBlankNode(child)) 
	    child = child->next; 

	  tmp = r_io_get_content(child, &err); 
	  if (tmp) 
	    {
	      net = r_net_address_new();
	      
	      if (IS_R_NET_ADDRESS(net))
		{
		  g_object_set(net, "url", tmp, 
			       "url-type", R_NET_ADDRESS_WEB, NULL);
		  r_card_add_net_address(card, net);
		  
		  g_free(tmp);
		  tmp = NULL;
		}
	      else
		return;
	    } 

	  if (child->next)
	    {
	      child = child->next; 
	      if (xmlIsBlankNode(child)) 
		child = child->next; 
	    }
	}
    }
}


void 
r_read_email(RCard *card, xmlNodePtr node) 
{
  RError err;
  RNetAddress* net;
  xmlNodePtr xmlemail = NULL;
  xmlNodePtr child;
  gchar *tmp; 
  
  xmlemail = r_io_get_node(node, (xmlChar*) "EmailAddresses"); 
  
  if (xmlemail)
    {
      // foreach child
      child = xmlemail->children;
      if (xmlIsBlankNode(child)) 
	child = child->next; 
      
      while (child)
	{
	  if (xmlIsBlankNode(child)) 
	    child = child->next; 
	  
	  tmp = r_io_get_content(child, &err); 
	  if (tmp)
	    {
	      net = r_net_address_new();
	      
	      if (IS_R_NET_ADDRESS(net))
		{
		  g_object_set(net, "url", tmp, 
			       "url-type", R_NET_ADDRESS_EMAIL, NULL);
		  r_card_add_net_address(card, net);
		  
		  g_free(tmp);
		  tmp = NULL;
		}
	      else
		return;
	    }
	  
	  // get next child
	  if (child->next)
	    {
	      child = child->next; 
	      if (xmlIsBlankNode(child)) 
		child = child->next; 
	    }
	}
    }
}


void 
r_read_address(RCard *card, xmlNodePtr node) 
{
  RAddress* ads;
  xmlNodePtr xmladdres = NULL;
  xmlNodePtr child;

  /*  
    <Address>
      <Street></Street>
      <StreetNumber></StreetNumber>
      <ZipCode></ZipCode>
      <City></City>
      <Province></Province>
      <State></State>
      <Country></Country>
    </Address>
  */

  xmladdres = r_io_get_node(node, (xmlChar*) "Address"); 
  
  if (xmladdres)
    {
      RError err;
      gchar *st, *num, *zip, *city, *prov, *state, *country;

      child = xmladdres->children;
      if (xmlIsBlankNode(child))
	child = child->next;
      
      ads = r_address_new();

      st      = r_io_get(child, "Street",       &err);
      num     = r_io_get(child, "StreetNumber", &err);
      zip     = r_io_get(child, "ZipCode",      &err);
      city    = r_io_get(child, "City",         &err);
      prov    = r_io_get(child, "Province",     &err);
      state   = r_io_get(child, "State",        &err);     
      country = r_io_get(child, "Country",      &err);
      
      g_object_set(G_OBJECT(ads), 
		   "address-type", R_ADDRESS_PREF,
		   "street", st, "street-number", num,
		   "city", city, "zip", zip, "province", prov,
		   "state", state, "country", country, NULL);

      r_card_add_address(card, ads);      
    }
}


void
r_read_telephone(RCard *card, xmlNodePtr cardnode)
{
  xmlNodePtr xmltel_child;
  xmlNodePtr xmltel = NULL;
  RError err;
  RTelephone* tel;
  RTelephoneType ttype;
  gchar* number;
  gchar* type;

  g_return_if_fail(IS_R_CARD(card));

  xmltel = r_io_get_node(cardnode, (xmlChar*) "TelephoneNumbers");

  if (xmltel)
    {
      xmltel_child = xmltel->children;
      if (xmlIsBlankNode(xmltel_child))
	xmltel_child = xmltel_child->next;

      while(xmltel_child)
	{
	  if (xmlIsBlankNode(xmltel_child))
	    xmltel_child = xmltel_child->next;
	  
	  number = r_io_get_content(xmltel_child, &err);
	  type   = r_io_get_prop(xmltel_child, "type", &err);

	  if (number)
	    {
	      tel = r_telephone_new();
	      
	      if(!IS_R_TELEPHONE(tel))
		{
		  g_warning("telephone obj get wrong type");
		  
		  return;
		}
	      
	      ttype = r_telephone_lookup_str2enum(type);
	      g_object_set(tel, "telephone-number", number, 
			   "telephone-type", ttype, NULL);

	      r_card_add_telephone(card, tel);

	      g_free(number);
	      g_free(type);
	    }
	  
	  xmltel_child = xmltel_child->next;
	  if (xmlIsBlankNode(xmltel_child)) 
	    xmltel_child = xmltel_child->next;
	} 
    }
}


void 
r_read_notes(RPersonalCard *card, xmlNodePtr cardnode)
{
  RError err;
  xmlNodePtr xmlnote = NULL;

  g_return_if_fail(IS_R_PERSONAL_CARD(card));

  xmlnote = r_io_get_node(cardnode, (xmlChar*) "Notes");
  
  if (xmlnote)
    {      
      RNotes* notes;
      gchar *day, *month, *year;
      gchar *partner, *other, *pubkey;
      gboolean bool, has_partner;

      notes = r_notes_new();

      if (!IS_R_NOTES(notes))
	return;

      has_partner = r_io_get_bool(xmlnote, "partner", &err);
      partner     = r_io_get(xmlnote, "PartnerName", &err);
      other       = r_io_get(xmlnote, "OtherNotes", &err);
      pubkey      = r_io_get(xmlnote, "PublicKey", &err);

      g_object_set(notes, "has-partner", has_partner, "partner-name", partner,
		   "other-notes", other, "pubkey", pubkey, NULL);

      g_free(partner);
      g_free(other);
      g_free(pubkey);
  
      bool = r_io_get_bool_from(xmlnote, "PartnerBirthday", 
				      "known", &err);
      r_io_get_calendar_from (xmlnote, "PartnerBirthday", 
				    &day, &month, &year, &err);
      if (bool)
	{
	  r_notes_set_know_birthday(notes, TRUE);
	  r_notes_set_birthday (notes, atoi(day), 
				      atoi(month), atoi(year));
	}

      bool = r_io_get_bool_from(xmlnote, "Anniversary", "known", &err);
      r_io_get_calendar_from (xmlnote, "Anniversary",&day, &month,
				    &year, &err);
      if (bool)
	{
	  r_notes_set_know_anniversary(notes, TRUE);
	  r_notes_set_anniversary(notes, atoi(day), 
					atoi(month), atoi(year)); 
	}

      r_personal_card_set_notes (card, notes);
    } 
}



void 
r_read_company (RCompanyCard* card, xmlNodePtr cardnode)
{
  RError err;
  xmlNodePtr xmlcompany = NULL;
  gchar* tmp;

  g_return_if_fail(IS_R_COMPANY_CARD(card));

  xmlcompany = r_io_get_node(cardnode, (xmlChar*) "Company");

  if (xmlcompany)
    {      
      tmp = r_io_get(xmlcompany, "CompanyName", &err);
      g_object_set(card, "company-name", tmp, NULL);
      g_free(tmp);
      
      tmp = r_io_get(xmlcompany, "Logo", &err);
      g_object_set(card, "company-logo", tmp, NULL);
      g_free(tmp);     

      tmp = r_io_get(xmlcompany, "VAT", &err);
      g_object_set(card, "company-vat", tmp, NULL);
      g_free(tmp);

      tmp = r_io_get(xmlcompany, "Notes", &err);
      g_object_set(card, "company-notes", tmp, NULL);
      g_free(tmp);
    }
}





/*    Public
*/

/*  open the file "filename"
    return ptr to xml doc or NULL on error
 */
xmlDocPtr 
r_open_doc(gchar *filename, RError* err)
{
  xmlDocPtr doc;

  *err = NO_FILENAME;
  g_return_val_if_fail(filename != NULL, NULL);
  
  /* read and parse file 
   */
  doc = xmlParseFile(filename);

  if (!doc)
    {
      *err = PARSING_DOC;
      
      return NULL;
    }  

  *err = NO_ERROR;
  return doc;
}


/*  get the file format of the r's doc    
*/
gint 
r_get_fileformat(xmlDocPtr doc, RError* err)
{
  gchar* tmp;
  gint ret;

  *err = NO_RUBRICA_DOC;
  g_return_val_if_fail(doc != NULL, -1);

  if(!xmlHasProp(doc->children, (xmlChar*) "fileformat"))
    {
      *err = UNKNOWN_FILEFORMAT;
      return 0;    
    }

  *err = NO_ERROR;
  tmp = (gchar*) xmlGetProp(doc->children, (xmlChar*) "fileformat");
  ret = atoi(tmp);
  xmlFree(tmp);

  return ret;
}


/*  get version.
    the version of r that make the file
*/
gchar*
r_get_version(xmlDocPtr doc, RError *err)
{
  gchar* ret = NULL;
  *err = NO_RUBRICA_DOC;
  g_return_val_if_fail(doc != NULL, NULL);
  
  if(!xmlHasProp(doc->children, (xmlChar*) "version"))
    {
      *err = UNKNOWN_VERSION;
      return NULL;
    }

  *err = NO_ERROR;
  ret = (gchar*) xmlGetProp(doc->children, (xmlChar*) "version");

  return ret;
}


/*  get the document type
    values: ADDRESSBOOK_TYPE_DOC
            ACCOUNT_TYPE_DOC         // removed, may be added in future
*/
gchar*
r_get_document_tpye(xmlDocPtr doc, RError *err)
{
  gchar* ret = NULL;

  *err = NO_RUBRICA_DOC;
  g_return_val_if_fail(doc != NULL, NULL);
  
  if(!xmlHasProp(doc->children, (xmlChar*) "doctype"))
    {
      *err = UNKNOWN_DOC;
      return NULL;
    }

  *err = NO_ERROR;
  ret  = (gchar*) xmlGetProp(doc->children, (xmlChar*) "doctype");

  return ret;
}


/*  check if file "filename" is a r doc
    return TRUE if xml doc is a r doc FALSE otherwise
*/
gboolean 
r_is_rubrica_doc(xmlDocPtr doc, RError *err)
{
  *err = NO_RUBRICA_DOC;
  g_return_val_if_fail(doc != NULL, FALSE);

  if (!doc->children || 
      !doc->children->name || 
      g_ascii_strcasecmp((gchar*) doc->children->name, 
			 (gchar*) "Rubrica") != 0)
    { 
      *err = UNKNOWN_DOC;
//      rubrica_dialog_error_loading_file(fname, RE_NO_RUBRICA_HEADER);  

      return FALSE;
    }  

  *err = NO_ERROR;

  return TRUE;
}


RPersonalCard* 
r_read_personal_xmlcard(xmlNodePtr xmlcard)
{
  RPersonalCard* card;

  card = r_personal_card_new();

  r_read_infos     (R_CARD(card), xmlcard);
  r_read_contact   (card, xmlcard);
  r_read_groups    (R_CARD(card), xmlcard);
  r_read_refs      (R_CARD(card), xmlcard);
  r_read_work      (card, xmlcard);
  r_read_addresses (R_CARD(card), xmlcard);
  r_read_net       (R_CARD(card), xmlcard);
  r_read_telephone (R_CARD(card), xmlcard);
  r_read_notes     (card, xmlcard);

  return card;
}

RCompanyCard *
r_read_company_xmlcard(xmlNodePtr xmlcard)
{
  RCompanyCard *card;

  card = r_company_card_new();

  r_read_infos     (R_CARD(card), xmlcard);
  r_read_company   (card, xmlcard);
  r_read_groups    (R_CARD(card), xmlcard);
  r_read_refs      (R_CARD(card), xmlcard);
  r_read_addresses (R_CARD(card), xmlcard);
  r_read_net       (R_CARD(card), xmlcard);
  r_read_telephone (R_CARD(card), xmlcard);

  return card;
}


RPersonalCard* 
r_read_old_personal_xmlcard(xmlNodePtr xmlcard)
{
  RPersonalCard* card = NULL;

  reading_old_card = TRUE;

  card = r_personal_card_new();
  
  r_read_infos     (R_CARD(card), xmlcard);   
  r_read_groups    (R_CARD(card), xmlcard);  
  r_read_contact   (card, xmlcard);  
  r_read_work      (card, xmlcard);  

  r_read_address   (R_CARD(card), xmlcard);    
  r_read_web       (R_CARD(card), xmlcard);    
  r_read_email     (R_CARD(card), xmlcard);    

  r_read_telephone (R_CARD(card), xmlcard);  
  r_read_notes     (card, xmlcard);  
    
  return card;
}

