/*
*  RAL -- Rubrica Addressbook Library
*  file: company.c
*  
*  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <time.h>
#include <glib.h>
#include <glib-object.h>

#include "types.h"
#include "card.h"
#include "company.h" 
#include "date.h"
#include "group.h"
#include "address.h"
#include "net.h"
#include "telephone.h"
#include "utils.h"



enum {
  PROP_0,
  COMPANY_NAME,
  COMPANY_LOGO,
  COMPANY_VAT,
  COMPANY_NOTES
};


struct _RCompanyCardPrivate {
  gchar* company_name;
  gchar* logo;
  gchar* vat;                /* VAT Value added tax (iva)/codice fiscale */ 
  gchar* notes;                 

  gboolean dispose_has_run;
};



static void r_company_card_class_init (RCompanyCardClass* klass);
static void r_company_card_init       (RCompanyCard* obj);

static void r_company_card_dispose    (RCompanyCard* obj);
static void r_company_card_finalize   (RCompanyCard* obj);

static void r_company_card_set_property (GObject* obj, guint property_id,
					 const GValue* value, 
					 GParamSpec* spec);
static void r_company_card_get_property (GObject* obj, guint property_id,
					 GValue* value, GParamSpec* spec);

static void   r_company_card_free (RCompanyCard* self); 

static gboolean r_company_card_search (RCompanyCard* self, const gchar* str);

RCompanyCard* r_company_card_copy (RCompanyCard* self);
RInfos*       r_company_get_infos (RCompanyCard* self);

static RCardClass* parent_class = NULL;



GType
r_company_card_get_type()
{
  static GType company_card_type = 0;
  
  if (!company_card_type)
    {
      static const GTypeInfo company_card_info =
	{
	  sizeof(RCompanyCardClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) r_company_card_class_init,
	  NULL,
	  NULL,
	  sizeof(RCompanyCard),
	  0,
	  (GInstanceInitFunc) r_company_card_init
	};

      company_card_type = g_type_register_static (R_CARD_TYPE,
						  "RCompanyCard",
						  &company_card_info, 0);
    }
  
  return company_card_type;
}


static void
r_company_card_class_init(RCompanyCardClass* klass)
{
  GObjectClass *class;
  GParamSpec* pspec;
  
  parent_class             = R_CARD_CLASS(klass);
  parent_class->free       = (gpointer) r_company_card_free;
  parent_class->copy       = (gpointer) r_company_card_copy;
  parent_class->search     = (gpointer) r_company_card_search;

  parent_class->get_infos  = (gpointer) r_company_get_infos;

  parent_class->print_card = NULL;
   
  class           = G_OBJECT_CLASS (klass);
  class->dispose  = (GObjectFinalizeFunc) r_company_card_dispose;
  class->finalize = (GObjectFinalizeFunc) r_company_card_finalize;

  class->set_property = r_company_card_set_property;
  class->get_property = r_company_card_get_property;
  
  /**
   * RCompanyCard:company-name
   *
   * the name of the company
   */
  pspec = g_param_spec_string("company-name",
			      "company name",
			      "the name of the company",
			      NULL,
			      G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(class, COMPANY_NAME, pspec);

  /**
   * RCompanyCard:notes
   *
   * notes about the company
   */
  pspec = g_param_spec_string("company-logo",
			      "logo",
			      "company's logo",
			      NULL, 
			      G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(class, COMPANY_LOGO, pspec);


  // FIXME: trovare nome migliore
  /**
   * RCompanyCard:pi
   *
   * Partita Iva (Codice Fiscale)
   * Note for translators and/or others: I need the english translation 
   * of "partita iva" and "codice fiscale". 
   * partita iva is the code used by companies to pay taxes (iva == VAT)
   * codice fiscale is the code associated to peoples
   */
  pspec = g_param_spec_string("company-vat",
			      "VAT or Taxpayer number",
			      "Value added tax or Taxpayer number",
			      NULL, 
			      G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(class, COMPANY_VAT, pspec);

  /**
   * RCompanyCard:notes
   *
   * notes about the company
   */
  pspec = g_param_spec_string("company-notes",
			      "notes",
			      "company's notes",
			      NULL, 
			      G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(class, COMPANY_NOTES, pspec);
}


static void
r_company_card_init(RCompanyCard* self)
{
  self->priv = g_new(RCompanyCardPrivate, 1);

  self->priv->company_name    = NULL;
  self->priv->logo            = NULL;
  self->priv->vat              = NULL;
  self->priv->notes           = NULL;

  self->priv->dispose_has_run = FALSE;
}



static void 
r_company_card_dispose (RCompanyCard* self)
{
  g_return_if_fail(IS_R_COMPANY_CARD(self));
    
  if (self->priv->dispose_has_run)
    return;

  self->priv->dispose_has_run = TRUE;
}


static void 
r_company_card_finalize (RCompanyCard* self)
{
  g_return_if_fail(IS_R_COMPANY_CARD(self));

  r_utils_free_string(self->priv->company_name);
  r_utils_free_string(self->priv->logo);  
  r_utils_free_string(self->priv->vat);  
  r_utils_free_string(self->priv->notes);
}



static void 
r_company_card_set_property (GObject* obj, guint property_id,
			     const GValue* value, GParamSpec* spec)
{
  RCompanyCard* self = (RCompanyCard*) obj;
  
  switch (property_id)
    {
    case COMPANY_NAME:
      g_free(self->priv->company_name);
      self->priv->company_name = g_value_dup_string(value);
      break;
    
    case COMPANY_LOGO:
      g_free(self->priv->logo);
      self->priv->logo = g_value_dup_string(value);
      break;

    case COMPANY_VAT:
      g_free(self->priv->vat);
      self->priv->vat = g_value_dup_string(value);
      break;
      
    case COMPANY_NOTES:
      g_free(self->priv->notes);
      self->priv->notes = g_value_dup_string(value);
      break;
      
    default:
      break;
    }
}


static void 
r_company_card_get_property (GObject* obj, guint property_id,
				   GValue* value, GParamSpec* spec)
{
  RCompanyCard* self = (RCompanyCard*) obj;
  
  switch (property_id)
    {
    case COMPANY_NAME:
      g_value_set_string(value, self->priv->company_name);
      break;
    
    case COMPANY_LOGO:
      g_value_set_string(value, self->priv->logo);
      break;
 
    case COMPANY_VAT:
      g_value_set_string(value, self->priv->vat);
      break;
      
    case COMPANY_NOTES:
      g_value_set_string(value, self->priv->notes);
      break;
      
    default:
      break;
    }  
}


/*  private functions
*/
static void
r_company_card_free(RCompanyCard* self)
{
  g_return_if_fail(IS_R_COMPANY_CARD(self));

  g_object_unref(G_OBJECT(self));            
}


RCompanyCard* 
r_company_card_copy (RCompanyCard* self)
{
  RCompanyCard* new;
  gchar *name, *logo, *vat, *notes;

  g_return_val_if_fail(IS_R_COMPANY_CARD(self), NULL);
 
  new = r_company_card_new();
  
  g_object_get(self, "company-name", &name, "company-logo", &logo,
	       "company-vat", &vat, "company-notes", &notes, NULL);
  g_object_set(new, "company-name", name, "company-logo", logo,
	       "company-vat", vat, "company-notes", notes, NULL); 

  return new;  
}


RInfos* 
r_company_get_infos (RCompanyCard* self)
{
  RInfos* infos;

 infos = g_malloc0(sizeof(RInfos));
 if (!infos)
   g_error("Unable to get enougth memory");
 
 g_object_get(self, "company-name", &infos->first, NULL);
 
 infos->last = infos->prof = infos->assignment = g_strdup(NULL);

 return infos;
}


/*  Public functions
*/

/**
 * r_company_card_new
 * 
 * create a new #RCompanyCard
 *
 * returns: a #RCompanyCard*
 */
RCompanyCard*
r_company_card_new(void)
{
  RCompanyCard* company;

  company = g_object_new(r_company_card_get_type(), 
			 "card-type", "company", NULL);
  
  return company;
}


static gboolean       
r_company_card_search (RCompanyCard* self, const gchar* str)
{
  gchar* name;
  gpointer data;
  
  g_object_get(R_CARD(self), "card-name", &name, NULL);
  
  if (g_strrstr (name, str)) // cerca l'ago (needle/str) 
                             // nel pagliaio (name/haystack)
    return TRUE;

  if (g_strrstr(self->priv->company_name, str))
    return TRUE;

  if (g_strrstr(self->priv->logo, str))
    return TRUE;

  if (g_strrstr(self->priv->vat, str))
    return TRUE;

  if (g_strrstr(self->priv->notes, str))
    return TRUE;


  /* search str in card's addresses */
  r_card_reset_address(R_CARD(self));
  data = r_card_get_address(R_CARD(self));
  for (; data; data = r_card_get_next_address(R_CARD(self)))
    if (r_address_search(R_ADDRESS(data), str))
      return TRUE;

  /* search str in card's net addresses */
  r_card_reset_net_address(R_CARD(self));
  data = r_card_get_net_address(R_CARD(self));
  for (; data; data = r_card_get_next_net_address(R_CARD(self)))
    if (r_net_address_search(R_NET_ADDRESS(data), str))
      return TRUE;  

  /* search str in card's telephone */
  r_card_reset_telephone(R_CARD(self));
  data = r_card_get_telephone(R_CARD(self));
  for (; data; data = r_card_get_next_telephone(R_CARD(self)))
    if (r_telephone_search(R_TELEPHONE(data), str))
      return TRUE;    

  return FALSE;  
}

