%% Copyright (C) 2016-2017 Colin B. Macdonald
%%
%% This file is part of OctSymPy.
%%
%% OctSymPy is free software; you can redistribute it and/or modify
%% it under the terms of the GNU General Public License as published
%% by the Free Software Foundation; either version 3 of the License,
%% or (at your option) any later version.
%%
%% This software is distributed in the hope that it will be useful,
%% but WITHOUT ANY WARRANTY; without even the implied warranty
%% of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
%% the GNU General Public License for more details.
%%
%% You should have received a copy of the GNU General Public
%% License along with this software; see the file COPYING.
%% If not, see <http://www.gnu.org/licenses/>.

%% -*- texinfo -*-
%% @documentencoding UTF-8
%% @defun sinhint (@var{x})
%% Numerical sinhint function.
%%
%% Example:
%% @example
%% @group
%% sinhint (1.1)
%%   @result{} ans = 1.1767
%% @end group
%% @end example
%%
%% @strong{Note} this function may be slow for large numbers of inputs.
%% This is because it is not a native double-precision implementation
%% but rather the numerical evaluation of the SymPy function
%% @code{Shi}.
%%
%% Note: this file is autogenerated: if you want to edit it, you might
%% want to make changes to 'generate_functions.py' instead.
%%
%% @seealso{@@sym/sinhint}
%% @end defun


function y = sinhint (x)
  if (nargin ~= 1)
    print_usage ();
  end
  cmd = { 'L = _ins[0]'
          'A = [complex(Shi(complex(x))) for x in L]'
          'return A,' };
  c = python_cmd (cmd, num2cell (x(:)));
  y = reshape (cell2mat (c), size (x));
end


%!test
%! x = 1.1;
%! y = sym(11)/10;
%! A = sinhint (x);
%! B = double (sinhint (y));
%! assert (A, B, -4*eps);

%!test
%! y = [2 3 sym(pi); exp(sym(1)) 5 6];
%! x = double (y);
%! A = sinhint (x);
%! B = double (sinhint (y));
%! assert (A, B, -4*eps);

%!test
%! % maple:
%! % > A := [1+2*I, -2 + 5*I, 100, 10*I, -1e-4 + 1e-6*I, -20 + I];
%! % > for a in A do evalf(Shi(a)) end do;
%! x = [1+2i; -2+5i; 100; 10i; -1e-4 + 1e-6*1i; -20-1i];
%! A = [  0.45769171128668800567 + 1.8332099215048436239*1i
%!        0.60888490416819746440 + 1.2482232175376056201*1i
%!        0.13577763724269399110e42
%!        1.6583475942188740493*1i
%!       -0.00010000000005553888891 + 0.10000000016666111119e-5*1i
%!       -0.74701205140887967022e7 - 0.10381447401236722090e8*1i ];
%! B = sinhint (x);
%! assert (A, B, -eps)
