package latexDraw.ui.dialog;

import java.awt.*;
import java.awt.event.ActionEvent;

import javax.swing.*;
import javax.swing.border.CompoundBorder;
import javax.swing.border.EmptyBorder;
import javax.swing.border.TitledBorder;
import javax.swing.event.ChangeEvent;

import latexDraw.figures.Ellipse;
import latexDraw.figures.Figure;
import latexDraw.figures.LaTeXDrawRectangle;
import latexDraw.lang.LaTeXDrawLang;
import latexDraw.ui.DrawPanel;
import latexDraw.util.LaTeXDrawPoint2D;


/** 
 * This class defines the frame which allows to change the parameters of ellipses and rectangles.<br>
 * <br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 01/20/06<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.0<br>
 */
public class ParametersEllipseRectangleFrame extends AbstractParametersFrame
{
	private static final long serialVersionUID = 1L;

	/** Allows to change the x-coordinates of the centre of the figure */
	protected JSpinner centerXField;
	
	/** Allows to change the y-coordinates of the centre of the figure */
	protected JSpinner centerYField;
	
	/** Allows to change the radius of the width of the ellipse */
	protected JSpinner widthRadiusField;
	
	/** Allows to change the radius of the height of the ellipse */
	protected JSpinner heightRadiusField;
	
	/** Allows to set if the rectangle will be round (only for rectangle */
	protected JCheckBox isRound;
	
	/** Allows to change the angle of the round corner. */
	protected JSpinner frameArcField;
	
	/** The panel containing to set the roundness of the figure. */
	protected JPanel roundPanel;
	
	
	/**
	 * The constructor by default
	 * @param p The drawPanel
	 */
	public ParametersEllipseRectangleFrame(Frame parent, DrawPanel p, boolean attachToMainFrame)
	{
		super(parent, p, attachToMainFrame);
		try
		{
	  		Dimension dim = Toolkit.getDefaultToolkit().getScreenSize();
	  		
	  		JTabbedPane tabbedPane = new JTabbedPane();
	  		JPanel pButton  = createButtonsPanel(this);
	  		JPanel pGeneral = new JPanel();
	  		pGeneral.setLayout(new BoxLayout(pGeneral, BoxLayout.Y_AXIS));
	  		
	  		JPanel coordP = new JPanel(new GridBagLayout());
	  		GridBagConstraints constraint = new GridBagConstraints();
	    	constraint.gridx = 0;  		constraint.gridy = 0;
	     	constraint.gridwidth = 1; 	constraint.gridheight = 1;
	     	constraint.weightx = 0.1; 	constraint.weighty = 0.1;
	     	constraint.fill = GridBagConstraints.NONE;
	     	constraint.anchor = GridBagConstraints.CENTER;
	     	coordP.add(new JLabel(LABEL_CENTER_X), constraint);
	  		
	     	SpinnerModel model = new SpinnerNumberModel(0,0,10000,0.1);
	     	centerXField = new JSpinner(model);
			centerXField.setEditor(new JSpinner.NumberEditor(centerXField, "0.0"));//$NON-NLS-1$
			centerXField.addChangeListener(this);
	     	centerXField.setName(LABEL_CENTER_X);
	     	constraint.gridx = 1;
	     	coordP.add(centerXField, constraint);

	    	constraint.gridx++;
	     	coordP.add(new JLabel(LABEL_CENTER_Y), constraint);
	  		
	     	model = new SpinnerNumberModel(0,0,10000,0.1);
	     	centerYField = new JSpinner(model);
	     	centerYField.setEditor(new JSpinner.NumberEditor(centerYField, "0.0"));//$NON-NLS-1$
			centerYField.addChangeListener(this);
	     	centerYField.setName(LABEL_CENTER_Y);
	     	constraint.gridx++;
	     	coordP.add(centerYField, constraint);
	     	
	    	constraint.gridx = 0;	constraint.gridy++;
	    	coordP.add(new JLabel(LABEL_WIDTH), constraint);
	  		
	     	model = new SpinnerNumberModel(0.1,0.1,10000,0.1);
	     	widthRadiusField = new JSpinner(model);
	     	widthRadiusField.setEditor(new JSpinner.NumberEditor(widthRadiusField, "0.0"));//$NON-NLS-1$
	     	widthRadiusField.addChangeListener(this);
	     	widthRadiusField.setName(LABEL_WIDTH);
	     	constraint.gridx++;
	     	coordP.add(widthRadiusField, constraint);
	   
	     	constraint.gridx++;
	     	coordP.add(new JLabel(LABEL_HEIGHT), constraint);
	  		
	     	model = new SpinnerNumberModel(0.1,0.1,10000,0.1);
	     	heightRadiusField = new JSpinner(model);
	     	heightRadiusField.setEditor(new JSpinner.NumberEditor(heightRadiusField, "0.0"));//$NON-NLS-1$
	     	heightRadiusField.addChangeListener(this);
	     	heightRadiusField.setName(LABEL_HEIGHT);
	     	constraint.gridx++;
	     	coordP.add(heightRadiusField, constraint);
	     	
	     	constraint.gridx = 0;	constraint.gridy++;
	     	coordP.add(new JLabel(LABEL_ROTATION), constraint);
	     	constraint.gridx++;
	     	coordP.add(rotationField, constraint);
	     	
	     	coordP.setBorder(new CompoundBorder(new TitledBorder(null, LABEL_POSITION_DIMENSIONS, 
							  TitledBorder.LEFT, TitledBorder.TOP), new EmptyBorder(0,0,0,0)));
	     	pGeneral.add(coordP);

	     	pGeneral.add(createBordersPanel(true));
	     	pGeneral.add(createDoubleLinePanel());
	     	pGeneral.add(createFillPanel());
	     	
	     	JPanel pOther = new JPanel();
	     	pOther.setLayout(new BoxLayout(pOther, BoxLayout.Y_AXIS));
	     	pOther.add(createHatchingPanel());
	     	pOther.add(createShadowPanel());
	     	pOther.add(createGradientPanel());
	     	
	     	roundPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
	     	
	     	isRound = new JCheckBox(LABEL_ISROUND);
	     	isRound.addActionListener(this);
	     	isRound.setActionCommand(LABEL_ISROUND);
	     	roundPanel.add(isRound);
	     	
	    	roundPanel.add(new JLabel(LABEL_ROUND_CORNER));
	    	
	     	model = new SpinnerNumberModel(LaTeXDrawRectangle.DEFAULT_FRAME_ARC, 0.01, 1,0.05);
			frameArcField = new JSpinner(model);
			frameArcField.setName(LABEL_ROUND_CORNER);
			frameArcField.setEditor(new JSpinner.NumberEditor(frameArcField, "0.00"));//$NON-NLS-1$
			frameArcField.addChangeListener(this);
			frameArcField.setEnabled(false);
	     	roundPanel.add(frameArcField);

	     	roundPanel.setBorder(new CompoundBorder(new TitledBorder(null, LaTeXDrawLang.getString1_7("AbstractParametersFrame.6"),  //$NON-NLS-1$
					  TitledBorder.LEFT, TitledBorder.TOP), new EmptyBorder(0,0,0,0)));
	     	pOther.add(roundPanel);
	     	
	     	tabbedPane.addTab(TITLE_TABBED_PANE_GENERAL, pGeneral);
	     	tabbedPane.addTab(TITLE_TABBED_PANE_OTHERS, pOther);
	     	BoxLayout bl = new BoxLayout(getContentPane(), BoxLayout.Y_AXIS);
	     	getContentPane().setLayout(bl);
	     	getContentPane().add(glimpsePanelZoom);
	     	getContentPane().add(tabbedPane);
			getContentPane().add(pButton);
	  		pack();
	  		setLocation(dim.width/2-getWidth()/2, dim.height/2-getHeight()/2);
			setVisible(false);

		}catch(Exception e)
		{
			e.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(e);
		}
	}
	
	
	
	@Override
	public void actionPerformed(ActionEvent e) 
	{
		try
		{
			Object o = e.getSource();
			Figure f = glimpsePanel.getGlimpseFigure();
			
			if(o instanceof JCheckBox)
			{
				String label = ((JCheckBox)o).getActionCommand();

				if(label.equals(LABEL_ISROUND))
				{
					setModified(true);
					((LaTeXDrawRectangle)f).setIsRound(isRound.isSelected());
					frameArcField.setEnabled(isRound.isSelected());
					glimpsePanel.repaint();
					return ;
				}
			}			
		}catch(Exception ex)
		{
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}
		
		super.actionPerformed(e);
	}

	
	
	@Override
	public void setFigureFrameField()
	{
		LaTeXDrawPoint2D NW = figure.getTheNWPoint(), SE = figure.getTheSEPoint();
		
		centerXField.setValue((NW.x+SE.x)/2.);
		centerYField.setValue((NW.y+SE.y)/2.);
		widthRadiusField.setValue((SE.x-NW.x)/2.);
		heightRadiusField.setValue((SE.y-NW.y)/2.);
		
		if(figure instanceof LaTeXDrawRectangle)
		{
			 isRound.setSelected(((LaTeXDrawRectangle)figure).isRound());
			 frameArcField.setEnabled(isRound.isSelected());
			 frameArcField.setValue(((LaTeXDrawRectangle)figure).getFrameArc());
			 roundPanel.setVisible(true);
		}
		else roundPanel.setVisible(false);
	}

	
	
	
	@Override
	public void setDefaultsValues()
	{
		super.setDefaultsValues();
		if(figure instanceof LaTeXDrawRectangle)
		{
			 isRound.setSelected(LaTeXDrawRectangle.DEFAULT_ISROUND);
			 frameArcField.setValue(LaTeXDrawRectangle.DEFAULT_FRAME_ARC);
			 ((LaTeXDrawRectangle)glimpsePanel.getGlimpseFigure()).setIsRound(LaTeXDrawRectangle.DEFAULT_ISROUND);
		}
	}
	
	
	
	@Override
	public void stateChanged(ChangeEvent e) 
	{
		try
		{
			Object o = e.getSource();
			
			Figure f = glimpsePanel.getGlimpseFigure();
			
			if(o instanceof JSpinner)
			{
				String name = ((JSpinner)o).getName();	
				
				if(name.equals(LABEL_ROUND_CORNER))
				{
					if(f instanceof LaTeXDrawRectangle)
					{
						((LaTeXDrawRectangle)f).setFrameArc(Double.valueOf(frameArcField.getValue().toString()).doubleValue());
						setModified(true);
						glimpsePanel.repaint();
					}
					return ;
				}
				
				LaTeXDrawPoint2D p1,p2,p3,p4;
				float widthRadius = Float.valueOf(widthRadiusField.getValue().toString()).floatValue();
				float heightRadius = Float.valueOf(heightRadiusField.getValue().toString()).floatValue();
				float x = Float.valueOf(centerXField.getValue().toString()).floatValue();
				float y = Float.valueOf(centerYField.getValue().toString()).floatValue();
				
				if(figure instanceof Ellipse)
				{
					Ellipse el = (Ellipse)f;
					p1 = el.getBordersPoint(0); p2 = el.getBordersPoint(1); 
					p3 = el.getBordersPoint(2); p4 = el.getBordersPoint(3);
				}
				else
				{
					LaTeXDrawRectangle r = (LaTeXDrawRectangle)f;
					p1 = r.getPoint(0); p2 = r.getPoint(1); 
					p3 = r.getPoint(2); p4 = r.getPoint(3);
				}
				
				if(name.equals(LABEL_CENTER_X))
				{
					p1.x = x - widthRadius;
					p2.x = x + widthRadius;
					p3.x = x - widthRadius;
					p4.x = x + widthRadius;
					f.updateShape();
					setModified(true);
					glimpsePanel.repaint();
					return ;
				}
				
				if(name.equals(LABEL_CENTER_Y))
				{
					p1.y = y - heightRadius;
					p2.y = y - heightRadius;
					p3.y = y + heightRadius;
					p4.y = y + heightRadius;
					f.updateShape();
					setModified(true);
					glimpsePanel.repaint();
					return ;
				}
				
				if(name.equals(LABEL_HEIGHT))
				{
					p1.y = y - heightRadius;
					p2.y = y - heightRadius;
					p3.y = y + heightRadius;
					p4.y = y + heightRadius;
					f.updateShape();
					setModified(true);
					glimpsePanel.repaint();
					return ;
				}
				
				if(name.equals(LABEL_WIDTH))
				{
					p1.x = x - widthRadius;
					p2.x = x + widthRadius;
					p3.x = x - widthRadius;
					p4.x = x + widthRadius;
					f.updateShape();
					setModified(true);
					glimpsePanel.repaint();
					return ;
				}
			}
		}catch(Exception ex)
		{
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}
		
		super.stateChanged(e);
	}
	
	
	
	@Override
	public void saveParameters()
	{
		super.saveParameters();
		
		LaTeXDrawPoint2D p1,p2,p3,p4;
		
		if(figure instanceof Ellipse)
		{
			Ellipse e = (Ellipse)figure;
			p1 = e.getBordersPoint(0); p2 = e.getBordersPoint(1); 
			p3 = e.getBordersPoint(2); p4 = e.getBordersPoint(3);
		}
		else
		{
			LaTeXDrawRectangle r = (LaTeXDrawRectangle)figure;
			p1 = r.getPoint(0); p2 = r.getPoint(1); 
			p3 = r.getPoint(2); p4 = r.getPoint(3);
			((LaTeXDrawRectangle)figure).setIsRound(isRound.isSelected());
			((LaTeXDrawRectangle)figure).setFrameArc(
					Double.valueOf(frameArcField.getValue().toString()).doubleValue());
		}
		
		float widthRadius = Float.valueOf(widthRadiusField.getValue().toString()).floatValue();
		float heightRadius = Float.valueOf(heightRadiusField.getValue().toString()).floatValue();
		float x = Float.valueOf(centerXField.getValue().toString()).floatValue();
		float y = Float.valueOf(centerYField.getValue().toString()).floatValue();
		
		p1.x = x - widthRadius;
		p2.x = x + widthRadius;
		p3.x = x - widthRadius;
		p4.x = x + widthRadius;

		p1.y = y - heightRadius;
		p2.y = y - heightRadius;
		p3.y = y + heightRadius;
		p4.y = y + heightRadius;

		if(figure instanceof Ellipse)
			((Ellipse)figure).getBorders().updateNSEWDelimitors();
		else ((LaTeXDrawRectangle)figure).updateNSEWDelimitors();
		
		figure.updateShape();
		
		drawPanel.getDraw().updateBorders();
		drawPanel.updateDraw(true);
	}
	
	

	@Override
	public void setVisible(boolean visible, Figure selected, boolean deleteOnCancel, boolean isFramedBox) 
	{
		try
		{
			if(!(selected instanceof Ellipse || selected instanceof LaTeXDrawRectangle))
				throw new IllegalArgumentException();
			
			if(isFramedBox)
			{
				centerXField.setEnabled(false);
				centerYField.setEnabled(false);
				widthRadiusField.setEnabled(false);
				heightRadiusField.setEnabled(false);
			}
			
			super.setVisible(visible, selected, deleteOnCancel, isFramedBox);
			
		}catch(Exception ex)
		{
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}	
	}
}
