/****************************************************************************
**
** Copyright (C) 2011 Nokia Corporation and/or its subsidiary(-ies).
** All rights reserved.
** Contact: Nokia Corporation (qt-info@nokia.com)
**
** This file is part of the QtGui module of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** GNU Lesser General Public License Usage
** This file may be used under the terms of the GNU Lesser General Public
** License version 2.1 as published by the Free Software Foundation and
** appearing in the file LICENSE.LGPL included in the packaging of this
** file. Please review the following information to ensure the GNU Lesser
** General Public License version 2.1 requirements will be met:
** http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain additional
** rights. These rights are described in the Nokia Qt LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU General
** Public License version 3.0 as published by the Free Software Foundation
** and appearing in the file LICENSE.GPL included in the packaging of this
** file. Please review the following information to ensure the GNU General
** Public License version 3.0 requirements will be met:
** http://www.gnu.org/copyleft/gpl.html.
**
** Other Usage
** Alternatively, this file may be used in accordance with the terms and
** conditions contained in a signed written agreement between you and Nokia.
**
**
**
**
**
** $QT_END_LICENSE$
**
****************************************************************************/

#ifndef KEXICOMPLETER_P_H
#define KEXICOMPLETER_P_H


//
//  W A R N I N G
//  -------------
//
// This file is not part of the Qt API.  It exists purely as an
// implementation detail.  This header file may change from version to
// version without notice, or even be removed.
//
// We mean it.
//

#ifndef QT_NO_COMPLETER

#include <QTreeView>
#include <QItemDelegate>
#include <QAbstractProxyModel>
#include <QPainter>
#include <QPointer>

#include "KexiCompleter.h"
#include "private/KexiAbstractItemModel_p.h"

class KexiCompletionModel;

class KexiCompleterPrivate
{
public:
    explicit KexiCompleterPrivate(KexiCompleter *q);
    ~KexiCompleterPrivate() { delete popup; }
    void init(QAbstractItemModel *model = 0);

    QPointer<QWidget> widget;
    KexiCompletionModel *proxy;
    QAbstractItemView *popup;
    KexiCompleter::CompletionMode mode;

    QString prefix;
    Qt::CaseSensitivity cs;
    bool substringCompletion;
    int role;
    int column;
    int maxVisibleItems;
    KexiCompleter::ModelSorting sorting;
    bool wrap;

    bool eatFocusOut;
    QRect popupRect;
    bool hiddenBecauseNoMatch;

    void showPopup(const QRect&);
    void _q_complete(QModelIndex, bool = false);
    void _q_completionSelected(const QItemSelection&);
    void _q_autoResizePopup();
    void _q_fileSystemModelDirectoryLoaded(const QString &path);
    void setCurrentIndex(QModelIndex, bool = true);

    KexiCompleter * const q;
};

class KexiIndexMapper
{
public:
    KexiIndexMapper() : v(false), f(0), t(-1) { }
    KexiIndexMapper(int f, int t) : v(false), f(f), t(t) { }
    explicit KexiIndexMapper(QVector<int> vec) : v(true), vector(vec), f(-1), t(-1) { }

    inline int count() const { return v ? vector.count() : t - f + 1; }
    inline int operator[] (int index) const { return v ? vector[index] : f + index; }
    inline int indexOf(int x) const { return v ? vector.indexOf(x) : ((t < f) ? -1 : x - f); }
    inline bool isValid() const { return !isEmpty(); }
    inline bool isEmpty() const { return v ? vector.isEmpty() : (t < f); }
    inline void append(int x) { Q_ASSERT(v); vector.append(x); }
    inline int first() const { return v ? vector.first() : f; }
    inline int last() const { return v ? vector.last() : t; }
    inline int from() const { Q_ASSERT(!v); return f; }
    inline int to() const { Q_ASSERT(!v); return t; }
    inline int cost() const { return vector.count()+2; }

private:
    bool v;
    QVector<int> vector;
    int f, t;
};

struct KexiMatchData {
    KexiMatchData() : exactMatchIndex(-1), partial(false) { }
    KexiMatchData(const KexiIndexMapper& indices, int em, bool p) :
        indices(indices), exactMatchIndex(em), partial(p) { }
    KexiIndexMapper indices;
    inline bool isValid() const { return indices.isValid(); }
    int  exactMatchIndex;
    bool partial;
};

class KexiCompletionEngine
{
public:
    typedef QMap<QString, KexiMatchData> CacheItem;
    typedef QMap<QModelIndex, CacheItem> Cache;

    explicit KexiCompletionEngine(KexiCompleterPrivate *c) : c(c), curRow(-1), cost(0) { }
    virtual ~KexiCompletionEngine() { }

    void filter(const QStringList &parts);

    KexiMatchData filterHistory();
    bool matchHint(QString, const QModelIndex&, KexiMatchData*);

    void saveInCache(QString, const QModelIndex&, const KexiMatchData&);
    bool lookupCache(QString part, const QModelIndex& parent, KexiMatchData *m);

    virtual void filterOnDemand(int) { }
    virtual KexiMatchData filter(const QString&, const QModelIndex&, int) = 0;

    int matchCount() const { return curMatch.indices.count() + historyMatch.indices.count(); }

    KexiMatchData curMatch, historyMatch;
    KexiCompleterPrivate *c;
    QStringList curParts;
    QModelIndex curParent;
    int curRow;

    Cache cache;
    int cost;
};

class QSortedModelEngine : public KexiCompletionEngine
{
public:
    explicit QSortedModelEngine(KexiCompleterPrivate *c) : KexiCompletionEngine(c) { }
    KexiMatchData filter(const QString&, const QModelIndex&, int);
    KexiIndexMapper indexHint(QString, const QModelIndex&, Qt::SortOrder);
    Qt::SortOrder sortOrder(const QModelIndex&) const;
};

class QUnsortedModelEngine : public KexiCompletionEngine
{
public:
    explicit QUnsortedModelEngine(KexiCompleterPrivate *c) : KexiCompletionEngine(c) { }

    void filterOnDemand(int);
    KexiMatchData filter(const QString&, const QModelIndex&, int);
private:
    int buildIndices(const QString& str, const QModelIndex& parent, int n,
                     const KexiIndexMapper& iv, KexiMatchData* m);
};

class KexiCompleterItemDelegate : public QItemDelegate
{
    Q_OBJECT
public:
    explicit KexiCompleterItemDelegate(QAbstractItemView *view)
        : QItemDelegate(view), view(view) { }
    void paint(QPainter *p, const QStyleOptionViewItem& opt, const QModelIndex& idx) const {
        QStyleOptionViewItem optCopy = opt;
        optCopy.showDecorationSelected = true;
        if (view->currentIndex() == idx)
            optCopy.state |= QStyle::State_HasFocus;
        QItemDelegate::paint(p, optCopy, idx);
    }

private:
    QAbstractItemView *view;
};

class KexiCompletionModelPrivate;

class KexiCompletionModel : public QAbstractProxyModel
{
    Q_OBJECT

public:
    KexiCompletionModel(KexiCompleterPrivate *c, QObject *parent);
    ~KexiCompletionModel();

    void createEngine();
    void setFiltered(bool);
    void filter(const QStringList& parts);
    int completionCount() const;
    int currentRow() const { return engine->curRow; }
    bool setCurrentRow(int row);
    QModelIndex currentIndex(bool) const;
    void resetModel();

    QModelIndex index(int row, int column, const QModelIndex & = QModelIndex()) const;
    int rowCount(const QModelIndex &index = QModelIndex()) const;
    int columnCount(const QModelIndex &index = QModelIndex()) const;
    bool hasChildren(const QModelIndex &parent = QModelIndex()) const;
    QModelIndex parent(const QModelIndex & = QModelIndex()) const { return QModelIndex(); }
    QVariant data(const QModelIndex& index, int role = Qt::DisplayRole) const;

    void setSourceModel(QAbstractItemModel *sourceModel);
    QModelIndex mapToSource(const QModelIndex& proxyIndex) const;
    QModelIndex mapFromSource(const QModelIndex& sourceIndex) const;

    KexiCompleterPrivate *c;
    QScopedPointer<KexiCompletionEngine> engine;
    bool showAll;

Q_SIGNALS:
    void rowsAdded();

public Q_SLOTS:
    void invalidate();
    void rowsInserted();
    void modelDestroyed();
private:
    KexiCompletionModelPrivate * const d;
};

class KexiCompletionModelPrivate : public KexiAbstractItemModelPrivate
{
    explicit KexiCompletionModelPrivate(KexiCompletionModel *q);
    virtual ~KexiCompletionModelPrivate() {  }
    virtual void _q_sourceModelDestroyed();
    KexiCompletionModel * const q;
    friend class KexiCompletionModel;
};

#endif // QT_NO_COMPLETER

#endif // KEXICOMPLETER_P_H
