#
# Copyright 2023 Centreon (http://www.centreon.com/)
#
# Centreon is a full-fledged industry-strength solution that meets
# the needs in IT infrastructure and application monitoring for
# service performance.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

package apps::grafana::api::mode::health;

use base qw(centreon::plugins::templates::counter);

use strict;
use warnings;
use centreon::plugins::templates::catalog_functions qw(catalog_status_threshold_ng);

sub custom_status_output {
    my ($self, %options) = @_;

    return sprintf(
        "database state is '%s' [grafana version: %s]", 
        $self->{result_values}->{state},
        $self->{result_values}->{version}
    );
}

sub set_counters {
    my ($self, %options) = @_;

    $self->{maps_counters_type} = [
        { name => 'health', type => 0, skipped_code => { -10 => 1 } }
    ];

    $self->{maps_counters}->{health} = [
        { label => 'status', type => 2, critical_default => '%{state} ne "ok"', set => {
                key_values => [ { name => 'state' }, { name => 'version' } ],
                closure_custom_output => $self->can('custom_status_output'),
                closure_custom_perfdata => sub { return 0; },
                closure_custom_threshold_check => \&catalog_status_threshold_ng
            }
        }
    ];
}

sub new {
    my ($class, %options) = @_;
    my $self = $class->SUPER::new(package => __PACKAGE__, %options, force_new_perfdata => 1);
    bless $self, $class;

    $options{options}->add_options(arguments => {
    });

    return $self;
}

sub manage_selection {
    my ($self, %options) = @_;

    my $result = $options{custom}->query(url_path => '/api/health');
    $self->{health} = {
        state => $result->{database},
        version => $result->{version}
    };
}

1;

__END__

=head1 MODE

Check health.

=over 8

=item B<--warning-status>

Set warning threshold for status.
Can used special variables like: %{state}, %{version}

=item B<--critical-status>

Set critical threshold for status (Default: '%{state} ne "ok"').
Can used special variables like: %{state}, %{version}

=back

=cut
