/*
 *  acm : an aerial combat simulator for X
 *  HSI tuner and HSI indicator module
 *  Copyright (C) 1991-1998  Riley Rainey
 *  Copyright (C) 2007  Umberto Salsi
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 dated June, 1991.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program;  if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave., Cambridge, MA 02139, USA.
 */

/**
 * HSI panel.
 * 
 * The HSI panel receives signals from VOR, DME, LOCATOR and GS stations,
 * then it feeds the data to the HSI display (course deviation indicator and
 * glide slope deviation indicator). For VOR+DME stations, implements also
 * the RNAV computer.
 * 
 * VOR, LOC and GS antennas all transmit a continuous signal that gives
 * angular informations. On the contrary, DME stations provide their response
 * to the transponder on board at random intervals of time, so the distance
 * has to be extrapolated in order to feed with continuous values the RNAV
 * calculator. Nevertheless DME and expecially CDI indications calculated
 * on WP by the RANV computer are quite unstable.
 * 
 *  BEWARE. In the following functions hsi_xxx_inc(u, step) usually the value
 *  of the step should be +1 (increment) or -1 (decrement) simply to indicate
 *  the direction of the change.  Every function internally implements an
 *  auto-repeat detection feature that speeds-up this otherwise very slow
 *  interface. If the auto-repeat gets detected, the increment actually
 *  applied gets multiplied by a factor that ranges from 2 up to 20.
 * 
 * @file
 */

#ifndef _hsi_h
#define _hsi_h

#include "pm.h"

#ifdef hsi_IMPORT
	#define EXTERN
#else
	#define EXTERN extern
#endif

/**
 * Enables the HSI for this particular viewer.  If u->hsi is NULL,
 * instantiate with a pointer to internal data.
 */
EXTERN void hsi_enable(viewer *u);

/**
 * Disables the HSI for this particular viewer.  If u->hsi is NULL,
 * instantiate with a pointer to internal data.
 */
EXTERN void hsi_disable(viewer *u);

/**
 * Release HSI data u->hsi.
 */
EXTERN void hsi_free(viewer *u);

/**
 * Loops between NAV1, NAV2, RNAV1, ... modes.
 */
EXTERN void hsi_switch_mode(viewer *u);

/**
 * Increment station frequency by step channels.  VOR/ILS/DME channels ranges
 * from 0 (108.00 MHz) up to 199 (117.95 MHz) by steps of 1 (0.05 MHz).
 */
EXTERN void hsi_frq_inc(viewer *u, int step);

/**
 * Increment OBS pointer by 'step' DEG.
 */
EXTERN void hsi_obs_inc(viewer *u, int step);

/**
 * Increment waypoint theta angle by 'step' DEG.
 */
void hsi_theta_inc(viewer *u, int step);

/**
 * Increment waypoint distance from VOR by 'step' tenth of NM.
 */
EXTERN void hsi_rho_inc(viewer *u, int step);

/**
 * Updates the internal state of the HSI. Does nothing if no HSI enabled in this
 * viewer.
 * @param u
 */
EXTERN void hsi_update(viewer * u);

/**
 * Draw HSI tuner panel (uses the TEWS panel). If no HSI active in this viewer,
 * do nothing.
 */
EXTERN void hsi_panel_draw(viewer * u);

/**
 * Draw HSI compass, OBS, CDI and GS (uses the radar panel).  If no HSI
 * active in this viewer, do nothing.
 */
EXTERN void hsi_draw(viewer * u);

/**
 * Return the currently selected OBS, or 0 if HSI not active.
 */
EXTERN int hsi_get_obs(viewer *u);

/**
 * Return in 'r' the radial (RAD) occupied by the aircraft in the VOR or
 * WP horizontal plane xy, with x aligned with the magnetic north at the
 * station location, y oriented magnetic east, -PI&le;r&le;PI.  Return FALSE
 * if no station tuned in the HSI or not a VOR and neither a WP.
 */
EXTERN _BOOL hsi_vor_radial(viewer *u, double *r);

/**
 * Return in 'r' the radial (RAD) occupied by the aircraft in the LOCATOR
 * plane xy, with x aligned with the LOCATOR bearing, i.e. r==0 means we
 * are aligned with LOC.  Return FALSE if no station tuned in the HSI or
 * not a LOCATOR.
 */
EXTERN _BOOL hsi_loc_radial(viewer *u, double *r);

/**
 * Return in 'dist' the distance (NM) from the DME/ILS-DME station or WP
 * currently selected in the HSI panel. Return FALSE if HSI not enabled or
 * no DME available.
 */
EXTERN _BOOL hsi_dme(viewer *u, double *dist);

/**
 * Return the offset angle (RAD) from the GS plane, positive if the aircraft
 * is too high. Return FALSE if no station tuned in the HSI or not an ILS.
 */
EXTERN _BOOL hsi_gs_offset(viewer *u, double *offset);

#undef EXTERN
#endif
