// Copyright (c) 2011 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "net/test/python_utils.h"

#include <memory>

#include "base/base_paths.h"
#include "base/command_line.h"
#include "base/environment.h"
#include "base/files/file_path.h"
#include "base/files/file_util.h"
#include "base/lazy_instance.h"
#include "base/logging.h"
#include "base/path_service.h"
#include "base/process/launch.h"
#include "base/strings/string_util.h"
#include "base/strings/utf_string_conversions.h"

#if defined(OS_MACOSX)
#include "base/mac/foundation_util.h"
#endif

const char kPythonPathEnv[] = "PYTHONPATH";
const char kPythonVirtualEnv[] = "VIRTUAL_ENV";

void ClearPythonPath() {
  std::unique_ptr<base::Environment> env(base::Environment::Create());
  env->UnSetVar(kPythonPathEnv);
}

void AppendToPythonPath(const base::FilePath& dir) {
  std::unique_ptr<base::Environment> env(base::Environment::Create());
  std::string old_path;
  std::string dir_path;
#if defined(OS_WIN)
  dir_path = base::WideToUTF8(dir.value());
#elif defined(OS_POSIX)
  dir_path = dir.value();
#endif
  if (!env->GetVar(kPythonPathEnv, &old_path)) {
    env->SetVar(kPythonPathEnv, dir_path.c_str());
  } else if (old_path.find(dir_path) == std::string::npos) {
    std::string new_path(old_path);
#if defined(OS_WIN)
    new_path.append(";");
#elif defined(OS_POSIX)
    new_path.append(":");
#endif
    new_path.append(dir_path.c_str());
    env->SetVar(kPythonPathEnv, new_path);
  }
}

bool GetPyProtoPath(base::FilePath* dir) {
  // Locate the Python code generated by the protocol buffers compiler.
  base::FilePath generated_code_dir;
  if (!PathService::Get(base::DIR_EXE, &generated_code_dir)) {
    LOG(ERROR) << "Can't find " << generated_code_dir.value();
    return false;
  }

#if defined(OS_MACOSX)
  if (base::mac::AmIBundled())
    generated_code_dir = generated_code_dir.DirName().DirName().DirName();
#endif

  // Used for GYP. TODO(jam): remove after GN conversion.
  const base::FilePath kPyProto(FILE_PATH_LITERAL("pyproto"));
  if (base::DirectoryExists(generated_code_dir.Append(kPyProto))) {
    *dir = generated_code_dir.Append(kPyProto);
    return true;
  }

  // Used for GN.
  const base::FilePath kGen(FILE_PATH_LITERAL("gen"));
  if (base::DirectoryExists(generated_code_dir.Append(kGen))) {
    *dir = generated_code_dir.Append(kGen);
    return true;
  }

  return false;
}

#if defined(OS_WIN)
struct PythonExePath {
  PythonExePath() {
    // This is test-only code, so CHECK with a subprocess invocation is ok.
    base::CommandLine command(base::FilePath(FILE_PATH_LITERAL("cmd")));
    command.AppendArg("/c");
    command.AppendArg("python");
    command.AppendArg("-c");
    command.AppendArg("import sys; print sys.executable");
    std::string output;
    CHECK(GetAppOutput(command, &output));
    // This does only work if cmd.exe doesn't use a non-US codepage.
    path_ = base::ASCIIToUTF16(output);
    TrimWhitespace(path_, base::TRIM_ALL, &path_);
  }
  base::string16 path_;
};
static base::LazyInstance<PythonExePath>::Leaky g_python_path;
#endif

bool IsInPythonVirtualEnv() {
  return base::Environment::Create()->HasVar(kPythonVirtualEnv);
}

bool GetPythonCommand(base::CommandLine* python_cmd) {
  DCHECK(python_cmd);

#if defined(OS_WIN)
  // Most developers have depot_tools in their path, which only has a
  // python.bat, not a python.exe.  Go through cmd to find the path to
  // the python executable.
  // (Don't just return a a "cmd /c python" command line, because then tests
  // that try to kill the python process will kill the cmd process instead,
  // which can cause flakiness.)
  python_cmd->SetProgram(base::FilePath(g_python_path.Get().path_));
#else
  python_cmd->SetProgram(base::FilePath(FILE_PATH_LITERAL("python")));
#endif

  // Launch python in unbuffered mode, so that python output doesn't mix with
  // gtest output in buildbot log files. See http://crbug.com/147368.
  python_cmd->AppendArg("-u");

  if (!IsInPythonVirtualEnv()) {
    // Prevent using system-installed libraries. Use hermetic versioned copies.
    python_cmd->AppendArg("-S");
  }

  return true;
}
