/*
 * Copyright (C) 2012 Michael Brown <mbrown@fensystems.co.uk>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

FILE_LICENCE ( GPL2_OR_LATER );

/** @file
 *
 * HMAC_DRBG tests
 *
 * These test vectors are provided by NIST as part of the
 * Cryptographic Toolkit Examples, downloadable from:
 *
 *    http://csrc.nist.gov/groups/ST/toolkit/documents/Examples/HMAC_DRBG.pdf
 *
 */

/* Forcibly enable assertions */
#undef NDEBUG

#include <assert.h>
#include <string.h>
#include <ipxe/hmac_drbg.h>
#include <ipxe/sha1.h>
#include <ipxe/sha256.h>
#include <ipxe/test.h>

/** Define inline expected data */
#define EXPECT(...) { __VA_ARGS__ }

/** An HMAC_DRBG instantiation test */
struct hmac_drbg_test_instantiate {
	/** Underlying hash algorithm */
	struct digest_algorithm *hash;
	/** Output block length */
	size_t out_len;
	/** Entropy */
	const void *entropy;
	/** Length of entropy */
	size_t entropy_len;
	/** Nonce */
	const void *nonce;
	/** Length of nonce */
	size_t nonce_len;
	/** Personalisation string */
	const void *personal;
	/** Length of personalisation string */
	size_t personal_len;
	/** Expected key */
	const void *expected_key;
	/** Length of expected key */
	size_t expected_key_len;
	/** Expected value */
	const void *expected_value;
	/** Length of expected value */
	size_t expected_value_len;
};

/**
 * Define an HMAC_DRBG instantiation test
 *
 * @v name		Test name
 * @v hmac_drbg		HMAC_DRBG algorithm
 * @v entropy_array	Entropy input
 * @v nonce_array	Nonce
 * @v personal_array	Personalisation string
 * @v key		Expected key
 * @v value		Expected value
 * @ret test		Instantiation test
 */
#define HMAC_DRBG_TEST_INSTANTIATE( name, hmac_drbg, entropy_array,	\
				    nonce_array, personal_array,	\
				    key, value )			\
	static const uint8_t name ## _key [] = key;			\
	static const uint8_t name ## _value [] = value;			\
	static struct hmac_drbg_test_instantiate name = {		\
		.hash = HMAC_DRBG_HASH ( hmac_drbg ),			\
		.out_len = HMAC_DRBG_OUTLEN_BYTES ( hmac_drbg ),	\
		.entropy = entropy_array,				\
		.entropy_len = sizeof ( entropy_array ),		\
		.nonce = nonce_array,					\
		.nonce_len = sizeof ( nonce_array ),			\
		.personal = personal_array,				\
		.personal_len = sizeof ( personal_array ),		\
		.expected_key = name ## _key,				\
		.expected_key_len = sizeof ( name ## _key ),		\
		.expected_value = name ## _value,			\
		.expected_value_len = sizeof ( name ## _value ),	\
	}

/**
 * Report instantiation test result
 *
 * @v state		HMAC_DRBG internal state
 * @v test		Instantiation test
 */
#define instantiate_ok( state, test ) do {				\
	struct {							\
		uint8_t entropy[(test)->entropy_len];			\
		uint8_t nonce[(test)->nonce_len];			\
	} __attribute__ (( packed )) entropy_nonce;			\
									\
	assert ( (test)->expected_key_len == (test)->out_len );		\
	assert ( (test)->expected_value_len == (test)->out_len );	\
	memcpy ( entropy_nonce.entropy, (test)->entropy,		\
		 sizeof ( entropy_nonce.entropy ) );			\
	memcpy ( entropy_nonce.nonce, (test)->nonce,			\
		 sizeof ( entropy_nonce.nonce ) );			\
	hmac_drbg_instantiate ( (test)->hash, (state), &entropy_nonce,	\
				sizeof ( entropy_nonce ),		\
				(test)->personal,			\
				(test)->personal_len );			\
	ok ( memcmp ( (state)->key, (test)->expected_key,		\
		      (test)->expected_key_len ) == 0 );		\
	ok ( memcmp ( (state)->value, (test)->expected_value,		\
		      (test)->expected_value_len ) == 0 );		\
	} while ( 0 )

/** An HMAC_DRBG reseed test */
struct hmac_drbg_test_reseed {
	/** Underlying hash algorithm */
	struct digest_algorithm *hash;
	/** Output block length */
	size_t out_len;
	/** Entropy */
	const void *entropy;
	/** Length of entropy */
	size_t entropy_len;
	/** Additional input */
	const void *additional;
	/** Length of additional_input */
	size_t additional_len;
	/** Expected key */
	const void *expected_key;
	/** Length of expected key */
	size_t expected_key_len;
	/** Expected value */
	const void *expected_value;
	/** Length of expected value */
	size_t expected_value_len;
};

/**
 * Define an HMAC_DRBG reseed test
 *
 * @v name		Test name
 * @v hmac_drbg		HMAC_DRBG algorithm
 * @v entropy_array	Entropy input
 * @v additional_array	Additional input
 * @v key		Expected key
 * @v value		Expected value
 * @ret test		Reseed test
 */
#define HMAC_DRBG_TEST_RESEED( name, hmac_drbg, entropy_array,		\
			       additional_array, key, value )		\
	static const uint8_t name ## _key [] = key;			\
	static const uint8_t name ## _value [] = value;			\
	static struct hmac_drbg_test_reseed name = {			\
		.hash = HMAC_DRBG_HASH ( hmac_drbg ),			\
		.out_len = HMAC_DRBG_OUTLEN_BYTES ( hmac_drbg ),	\
		.entropy = entropy_array,				\
		.entropy_len = sizeof ( entropy_array ),		\
		.additional = additional_array,				\
		.additional_len = sizeof ( additional_array ),		\
		.expected_key = name ## _key,				\
		.expected_key_len = sizeof ( name ## _key ),		\
		.expected_value = name ## _value,			\
		.expected_value_len = sizeof ( name ## _value ),	\
	}

/**
 * Report reseed test result
 *
 * @v state		HMAC_DRBG internal state
 * @v test		Reseed test
 */
#define reseed_ok( state, test ) do {					\
	assert ( (test)->expected_key_len == (test)->out_len );		\
	assert ( (test)->expected_value_len == (test)->out_len );	\
	hmac_drbg_reseed ( (test)->hash, (state), (test)->entropy,	\
			   (test)->entropy_len, (test)->additional,	\
			   (test)->additional_len );			\
	ok ( memcmp ( (state)->key, (test)->expected_key,		\
		      (test)->expected_key_len ) == 0 );		\
	ok ( memcmp ( (state)->value, (test)->expected_value,		\
		      (test)->expected_value_len ) == 0 );		\
	} while ( 0 )

/** An HMAC_DRBG generation test */
struct hmac_drbg_test_generate {
	/** Underlying hash algorithm */
	struct digest_algorithm *hash;
	/** Output block length */
	size_t out_len;
	/** Additional input */
	const void *additional;
	/** Length of additional_input */
	size_t additional_len;
	/** Expected key */
	const void *expected_key;
	/** Length of expected key */
	size_t expected_key_len;
	/** Expected value */
	const void *expected_value;
	/** Length of expected value */
	size_t expected_value_len;
	/** Expected pseudorandom data */
	const void *expected_data;
	/** Length of data */
	size_t expected_data_len;
};

/**
 * Define an HMAC_DRBG generation test
 *
 * @v name		Test name
 * @v hmac_drbg		HMAC_DRBG algorithm
 * @v additional_array	Additional input
 * @v key		Expected key
 * @v value		Expected value
 * @v data		Expected pseudorandom data
 * @ret test		Generation test
 */
#define HMAC_DRBG_TEST_GENERATE( name, hmac_drbg, additional_array,	\
				 key, value, data )			\
	static const uint8_t name ## _key [] = key;			\
	static const uint8_t name ## _value [] = value;			\
	static const uint8_t name ## _data [] = data;			\
	static struct hmac_drbg_test_generate name = {			\
		.hash = HMAC_DRBG_HASH ( hmac_drbg ),			\
		.out_len = HMAC_DRBG_OUTLEN_BYTES ( hmac_drbg ),	\
		.additional = additional_array,				\
		.additional_len = sizeof ( additional_array ),		\
		.expected_key = name ## _key,				\
		.expected_key_len = sizeof ( name ## _key ),		\
		.expected_value = name ## _value,			\
		.expected_value_len = sizeof ( name ## _value ),	\
		.expected_data = name ## _data,				\
		.expected_data_len = sizeof ( name ## _data ),		\
	}

/**
 * Report generation test result
 *
 * @v state		HMAC_DRBG internal state
 * @v test		Generation test
 */
#define generate_ok( state, test ) do {					\
	uint8_t data[ (test)->expected_data_len ];			\
	int rc;								\
									\
	assert ( (test)->expected_key_len == (test)->out_len );		\
	assert ( (test)->expected_value_len == (test)->out_len );	\
	rc = hmac_drbg_generate ( (test)->hash, (state),		\
				  (test)->additional,			\
				  (test)->additional_len,		\
				  data, sizeof ( data ) );		\
	ok ( rc == 0 );							\
	ok ( memcmp ( (state)->key, (test)->expected_key,		\
		      (test)->expected_key_len ) == 0 );		\
	ok ( memcmp ( (state)->value, (test)->expected_value,		\
		      (test)->expected_value_len ) == 0 );		\
	ok ( memcmp ( data, (test)->expected_data,			\
		      (test)->expected_data_len ) == 0 );		\
	} while ( 0 )

/** An HMAC_DRBG generation failure test */
struct hmac_drbg_test_generate_fail {
	/** Underlying hash algorithm */
	struct digest_algorithm *hash;
	/** Additional input */
	const void *additional;
	/** Length of additional_input */
	size_t additional_len;
	/** Length of requested data */
	size_t requested_len;
};

/**
 * Define an HMAC_DRBG generation failure test
 *
 * @v name		Test name
 * @v hmac_drbg		HMAC_DRBG algorithm
 * @v additional_array	Additional input
 * @ret test		Generation failure test
 */
#define HMAC_DRBG_TEST_GENERATE_FAIL( name, hmac_drbg,			\
				      additional_array, len )		\
	static struct hmac_drbg_test_generate_fail name = {		\
		.hash = HMAC_DRBG_HASH ( hmac_drbg ),			\
		.additional = additional_array,				\
		.additional_len = sizeof ( additional_array ),		\
		.requested_len = len,					\
	}

/**
 * Report generation failure test result
 *
 * @v state		HMAC_DRBG internal state
 * @v test		Generation failure test
 */
#define generate_fail_ok( state, test ) do {				\
	uint8_t data[ (test)->requested_len ];				\
	int rc;								\
									\
	rc = hmac_drbg_generate ( (test)->hash, (state),		\
				  (test)->additional,			\
				  (test)->additional_len, data,		\
				  sizeof ( data ) );			\
	ok ( rc != 0 );							\
	} while ( 0 )

/** "EntropyInput" */
static const uint8_t entropy_input[] = {
	0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
	0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
	0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23,
	0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
	0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36
};

/** "Nonce" for SHA-1 */
static const uint8_t nonce_sha1[] = {
	0x20, 0x21, 0x22, 0x23, 0x24
};

/** "Nonce" for SHA-256 */
static const uint8_t nonce_sha256[] = {
	0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27
};

/** "EntropyInput1 (for Reseed1) */
static const uint8_t entropy_input_1[] = {
	0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b,
	0x8c, 0x8d, 0x8e, 0x8f, 0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
	0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 0xa0, 0xa1, 0xa2, 0xa3,
	0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
	0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6
};

/** "EntropyInput2 (for Reseed2) */
static const uint8_t entropy_input_2[] = {
	0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb,
	0xcc, 0xcd, 0xce, 0xcf, 0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
	0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 0xe0, 0xe1, 0xe2, 0xe3,
	0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
	0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6
};

/** "PersonalizationString = <empty>" */
static const uint8_t personalisation_string_empty[] = {};

/** "PersonalizationString" */
static const uint8_t personalisation_string[] = {
	0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b,
	0x4c, 0x4d, 0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
	0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f, 0x60, 0x61, 0x62, 0x63,
	0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
	0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76
};

/** "AdditionalInput = <empty>" */
static const uint8_t additional_input_empty[] = {};

/** "AdditionalInput1" */
static const uint8_t additional_input_1[] = {
	0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b,
	0x6c, 0x6d, 0x6e, 0x6f, 0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
	0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 0x80, 0x81, 0x82, 0x83,
	0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
	0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96
};

/** "AdditionalInput2" */
static const uint8_t additional_input_2[] = {
	0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab,
	0xac, 0xad, 0xae, 0xaf, 0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7,
	0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 0xc0, 0xc1, 0xc2, 0xc3,
	0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
	0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6
};

/** SHA-1 Test 1 : Instantiation */
HMAC_DRBG_TEST_INSTANTIATE ( sha1_instantiate_1, HMAC_DRBG_SHA1,
	entropy_input, nonce_sha1, personalisation_string_empty,
	EXPECT ( 0xab, 0x16, 0x0d, 0xd2, 0x1c, 0x30, 0x98, 0x0c, 0xa3, 0xca,
		 0x5a, 0x9c, 0x77, 0xb7, 0xbd, 0xf0, 0x50, 0xe6, 0x4e, 0xe9 ),
	EXPECT ( 0x61, 0x44, 0x99, 0xea, 0x98, 0x0c, 0xfb, 0x3d, 0xaa, 0x2c,
		 0xa8, 0x6d, 0x65, 0xa4, 0x6b, 0xf4, 0x48, 0x8d, 0x8c, 0xc5 ) );

/** SHA-1 Test 1.1 : First call to Generate */
HMAC_DRBG_TEST_GENERATE ( sha1_generate_1_1, HMAC_DRBG_SHA1,
	additional_input_empty,
	EXPECT ( 0x7b, 0xb1, 0x80, 0x28, 0xe0, 0x1d, 0x03, 0x42, 0xdf, 0x4f,
		 0x54, 0xda, 0x51, 0x22, 0xfa, 0x5f, 0x2c, 0x3a, 0x05, 0xe4 ),
	EXPECT ( 0x2f, 0x89, 0x4f, 0x28, 0xcc, 0x2f, 0x53, 0x82, 0x96, 0x40,
		 0x64, 0x3a, 0xd1, 0x7b, 0x84, 0xb0, 0xcd, 0x3c, 0x79, 0x79 ),
	EXPECT ( 0x5a, 0x7d, 0x3b, 0x44, 0x9f, 0x48, 0x1c, 0xb3, 0x8d, 0xf7,
		 0x9a, 0xd2, 0xb1, 0xfc, 0xc0, 0x1e, 0x57, 0xf8, 0x13, 0x5e,
		 0x8c, 0x0b, 0x22, 0xcd, 0x06, 0x30, 0xbf, 0xb0, 0x12, 0x7f,
		 0xb5, 0x40, 0x8c, 0x8e, 0xfc, 0x17, 0xa9, 0x29, 0x89, 0x6e ) );

/** SHA-1 Test 1.2 : Second call to Generate */
HMAC_DRBG_TEST_GENERATE ( sha1_generate_1_2, HMAC_DRBG_SHA1,
	additional_input_empty,
	EXPECT ( 0x3d, 0x4d, 0x73, 0x77, 0xe9, 0x17, 0x2a, 0xaf, 0xa7, 0x76,
		 0xb0, 0xdd, 0xcb, 0x89, 0x42, 0x00, 0x4a, 0x44, 0xb7, 0xfd ),
	EXPECT ( 0x1a, 0x26, 0xbd, 0x9b, 0xfc, 0x97, 0x44, 0xbd, 0x29, 0xf6,
		 0xae, 0xbe, 0x24, 0x37, 0xe2, 0x09, 0xf1, 0xf7, 0x16, 0x25 ),
	EXPECT ( 0x82, 0xcf, 0x77, 0x2e, 0xc3, 0xe8, 0x4b, 0x00, 0xfc, 0x74,
		 0xf5, 0xdf, 0x10, 0x4e, 0xfb, 0xfb, 0x24, 0x28, 0x55, 0x4e,
		 0x9c, 0xe3, 0x67, 0xd0, 0x3a, 0xea, 0xde, 0x37, 0x82, 0x7f,
		 0xa8, 0xe9, 0xcb, 0x6a, 0x08, 0x19, 0x61, 0x15, 0xd9, 0x48 ) );

/** SHA-1 Test 2 : Instantiation */
#define sha1_instantiate_2 sha1_instantiate_1

/** SHA-1 Test 2.1 : First call to Generate */
HMAC_DRBG_TEST_GENERATE ( sha1_generate_2_1, HMAC_DRBG_SHA1,
	additional_input_1,
	EXPECT ( 0x3a, 0x06, 0x2e, 0x6b, 0x79, 0xfe, 0x70, 0xdb, 0xff, 0xeb,
		 0x3a, 0x2b, 0x6b, 0xe8, 0x03, 0x23, 0xf7, 0xd6, 0x74, 0xc5 ),
	EXPECT ( 0xbd, 0x36, 0x31, 0x28, 0xbf, 0x58, 0x0d, 0x7a, 0x54, 0x42,
		 0x9d, 0xdd, 0x58, 0xe8, 0x19, 0x3b, 0x98, 0x43, 0xbd, 0x2b ),
	EXPECT ( 0xc7, 0xaa, 0xac, 0x58, 0x3c, 0x6e, 0xf6, 0x30, 0x07, 0x14,
		 0xc2, 0xcc, 0x5d, 0x06, 0xc1, 0x48, 0xcf, 0xfb, 0x40, 0x44,
		 0x9a, 0xd0, 0xbb, 0x26, 0xfa, 0xc0, 0x49, 0x7b, 0x5c, 0x57,
		 0xe1, 0x61, 0xe3, 0x66, 0x81, 0xbc, 0xc9, 0x30, 0xce, 0x80 ) );

/** SHA-1 Test 2.2 : Second call to Generate */
HMAC_DRBG_TEST_GENERATE ( sha1_generate_2_2, HMAC_DRBG_SHA1,
	additional_input_2,
	EXPECT ( 0x8a, 0xd7, 0xe3, 0x47, 0x72, 0xb5, 0xfc, 0x7c, 0x3b, 0x3b,
		 0x27, 0x62, 0x4f, 0x0b, 0x91, 0x77, 0x6a, 0x8a, 0x71, 0x12 ),
	EXPECT ( 0xd7, 0x13, 0x76, 0xa4, 0x6d, 0x76, 0x4b, 0x17, 0xc3, 0xb7,
		 0x39, 0x34, 0x7b, 0x38, 0x4e, 0x51, 0x51, 0xe8, 0x7e, 0x88 ),
	EXPECT ( 0x6e, 0xbd, 0x2b, 0x7b, 0x5e, 0x0a, 0x2a, 0xd7, 0xa2, 0x4b,
		 0x1b, 0xf9, 0xa1, 0xdb, 0xa4, 0x7d, 0x43, 0x27, 0x17, 0x19,
		 0xb9, 0xc3, 0x7b, 0x7f, 0xe8, 0x1b, 0xa9, 0x40, 0x45, 0xa1,
		 0x4a, 0x7c, 0xb5, 0x14, 0xb4, 0x46, 0x66, 0x6e, 0xa5, 0xa7 ) );

/** SHA-1 Test 3 : Instantiation */
HMAC_DRBG_TEST_INSTANTIATE ( sha1_instantiate_3, HMAC_DRBG_SHA1,
	entropy_input, nonce_sha1, personalisation_string,
	EXPECT ( 0xb7, 0xd9, 0x66, 0xd7, 0x0d, 0x4e, 0x27, 0xa7, 0xfa, 0x83,
		 0x8f, 0x7d, 0x61, 0x12, 0x6c, 0x0e, 0xdc, 0x84, 0x76, 0x1c ),
	EXPECT ( 0xda, 0xb2, 0xa7, 0x18, 0x83, 0xf1, 0x00, 0x5c, 0x5d, 0xd0,
		 0x39, 0x32, 0x4d, 0x3c, 0x36, 0x4d, 0x6e, 0x18, 0xf9, 0x54 ) );

/** SHA-1 Test 3.1 : First call to Generate */
HMAC_DRBG_TEST_GENERATE ( sha1_generate_3_1, HMAC_DRBG_SHA1,
	additional_input_empty,
	EXPECT ( 0x87, 0xd3, 0x82, 0x8b, 0xe0, 0x3a, 0x80, 0x7d, 0xd3, 0x40,
		 0x29, 0x41, 0xbe, 0xd6, 0xde, 0x98, 0x6e, 0xe7, 0xa2, 0x86 ),
	EXPECT ( 0x6a, 0xe1, 0xd0, 0x08, 0x6f, 0x53, 0xb1, 0xb7, 0x63, 0xa4,
		 0x51, 0x5b, 0x19, 0x06, 0xfe, 0xe4, 0x76, 0x61, 0xfd, 0x47 ),
	EXPECT ( 0xb3, 0xbd, 0x05, 0x24, 0x6c, 0xba, 0x12, 0xa6, 0x47, 0x35,
		 0xa4, 0xe3, 0xfd, 0xe5, 0x99, 0xbc, 0x1b, 0xe3, 0x0f, 0x43,
		 0x9b, 0xd0, 0x60, 0x20, 0x8e, 0xea, 0x7d, 0x71, 0xf9, 0xd1,
		 0x23, 0xdf, 0x47, 0xb3, 0xce, 0x06, 0x9d, 0x98, 0xed, 0xe6 ) );

/** SHA-1 Test 3.2 : Second call to Generate */
HMAC_DRBG_TEST_GENERATE ( sha1_generate_3_2, HMAC_DRBG_SHA1,
	additional_input_empty,
	EXPECT ( 0x26, 0xab, 0xbf, 0x54, 0xb2, 0x8b, 0x93, 0xff, 0x90, 0x08,
		 0x67, 0x0e, 0xbf, 0xee, 0x86, 0xcd, 0xd7, 0x22, 0x8e, 0xd5 ),
	EXPECT ( 0xe9, 0x25, 0x47, 0x29, 0xe0, 0x02, 0x04, 0xa1, 0xb6, 0xc0,
		 0x21, 0x58, 0xa6, 0xc7, 0x27, 0x86, 0x47, 0x14, 0xf1, 0xf7 ),
	EXPECT ( 0xb5, 0xda, 0xda, 0x38, 0x0e, 0x28, 0x72, 0xdf, 0x93, 0x5b,
		 0xca, 0x55, 0xb8, 0x82, 0xc8, 0xc9, 0x37, 0x69, 0x02, 0xab,
		 0x63, 0x97, 0x65, 0x47, 0x2b, 0x71, 0xac, 0xeb, 0xe2, 0xea,
		 0x8b, 0x1b, 0x6b, 0x49, 0x62, 0x9c, 0xb6, 0x73, 0x17, 0xe0 ) );

/** SHA-1 Test 4 : Instantiation */
#define sha1_instantiate_4 sha1_instantiate_3

/** SHA-1 Test 4.1 : First call to Generate */
HMAC_DRBG_TEST_GENERATE ( sha1_generate_4_1, HMAC_DRBG_SHA1,
	additional_input_1,
	EXPECT ( 0x17, 0xa5, 0xd7, 0x9f, 0x07, 0x67, 0x87, 0x6f, 0x3a, 0x45,
		 0xe0, 0xc9, 0xc3, 0x3e, 0xc8, 0x8b, 0x03, 0xce, 0xea, 0x13 ),
	EXPECT ( 0x4d, 0x2f, 0x3b, 0xc7, 0x77, 0x50, 0x5c, 0x45, 0xf7, 0xe1,
		 0x7d, 0xcd, 0x3d, 0x86, 0xbf, 0x37, 0x9c, 0xb6, 0x02, 0x5e ),
	EXPECT ( 0x1f, 0x8f, 0xec, 0x7b, 0xc7, 0xcf, 0xa9, 0xa8, 0x80, 0x34,
		 0x5d, 0x28, 0x0b, 0x13, 0xc6, 0x32, 0xb8, 0x52, 0x77, 0x0a,
		 0x6d, 0xfc, 0x30, 0x2e, 0xad, 0x4c, 0xe3, 0xf5, 0x54, 0xc7,
		 0x9b, 0x0d, 0x44, 0x23, 0x9e, 0xba, 0x56, 0xa7, 0xea, 0x2d ) );

/** SHA-1 Test 4.2 : Second call to Generate */
HMAC_DRBG_TEST_GENERATE ( sha1_generate_4_2, HMAC_DRBG_SHA1,
	additional_input_2,
	EXPECT ( 0x07, 0x9b, 0x57, 0xd9, 0x40, 0x6e, 0x11, 0xc2, 0xf8, 0x7c,
		 0x8c, 0x82, 0x8c, 0x8c, 0x6f, 0xa7, 0x6e, 0x40, 0xea, 0x01 ),
	EXPECT ( 0xa6, 0x54, 0xfe, 0x72, 0xf8, 0xa7, 0x7b, 0xb8, 0xf0, 0x3d,
		 0xff, 0x07, 0xc7, 0x9a, 0x51, 0x53, 0x00, 0x9e, 0xdd, 0xda ),
	EXPECT ( 0xaf, 0x97, 0xcd, 0xe1, 0xe8, 0xab, 0x32, 0x2a, 0x2e, 0xac,
		 0xa8, 0xe6, 0xf4, 0xe5, 0xbf, 0x78, 0xa1, 0x1b, 0xde, 0xf7,
		 0xdc, 0x91, 0x21, 0x5d, 0x44, 0xb1, 0x07, 0xb4, 0xd5, 0xa7,
		 0x79, 0x01, 0x59, 0x25, 0x09, 0x76, 0x52, 0x80, 0xf9, 0x69 ) );

/** SHA-1 Test 5 : Instantiation */
#define sha1_instantiate_5 sha1_instantiate_1

/** SHA-1 Test 5.1 : First call to Generate */
HMAC_DRBG_TEST_GENERATE_FAIL ( sha1_generate_fail_5_1, HMAC_DRBG_SHA1,
	additional_input_empty, ( 320 / 8 ) );

/** SHA-1 Test 5.2 : Reseed */
HMAC_DRBG_TEST_RESEED ( sha1_reseed_5_2, HMAC_DRBG_SHA1,
	entropy_input_1, additional_input_empty,
	EXPECT ( 0xcd, 0x4c, 0xab, 0x38, 0xc8, 0xad, 0x65, 0x71, 0x22, 0xbf,
		 0x5d, 0x3d, 0x00, 0xd0, 0xac, 0x9b, 0x13, 0xd6, 0x29, 0xbb ),
	EXPECT ( 0xf6, 0x60, 0xe2, 0x3e, 0x91, 0x00, 0x6b, 0x62, 0xc6, 0x54,
		 0x3a, 0xb1, 0x34, 0x4d, 0x23, 0xa3, 0x1a, 0xb4, 0xcf, 0x2c ) );

/** SHA-1 Test 5.3 : Retried first call to Generate */
HMAC_DRBG_TEST_GENERATE ( sha1_generate_5_3, HMAC_DRBG_SHA1,
	additional_input_empty,
	EXPECT ( 0x58, 0x7f, 0xd8, 0x21, 0xef, 0x6c, 0x9d, 0xa4, 0xa8, 0x3c,
		 0x19, 0x21, 0x1f, 0x10, 0x56, 0xca, 0xcd, 0x23, 0xfc, 0x1a ),
	EXPECT ( 0x84, 0x8f, 0xd1, 0x4c, 0x13, 0xb7, 0xea, 0x93, 0x72, 0x0c,
		 0xcf, 0xde, 0x71, 0xf2, 0xf6, 0x44, 0x39, 0xdb, 0x79, 0x5d ),
	EXPECT ( 0xfe, 0xc4, 0x59, 0x7f, 0x06, 0xa3, 0xa8, 0xcc, 0x85, 0x29,
		 0xd5, 0x95, 0x57, 0xb9, 0xe6, 0x61, 0x05, 0x38, 0x09, 0xc0,
		 0xbc, 0x0e, 0xfc, 0x28, 0x2a, 0xbd, 0x87, 0x60, 0x5c, 0xc9,
		 0x0c, 0xba, 0x9b, 0x86, 0x33, 0xdc, 0xb1, 0xda, 0xe0, 0x2e ) );

/** SHA-1 Test 5.4 : Second call to Generate */
HMAC_DRBG_TEST_GENERATE_FAIL ( sha1_generate_fail_5_4, HMAC_DRBG_SHA1,
	additional_input_empty, ( 320 / 8 ) );

/** SHA-1 Test 5.5 : Reseed */
HMAC_DRBG_TEST_RESEED ( sha1_reseed_5_5, HMAC_DRBG_SHA1,
	entropy_input_2, additional_input_empty,
	EXPECT ( 0xdb, 0xa1, 0xcf, 0xf4, 0x87, 0x95, 0x46, 0xa0, 0x38, 0xa5,
		 0x59, 0xb2, 0xa2, 0x4d, 0xf2, 0xc0, 0x30, 0x08, 0x9a, 0x41 ),
	EXPECT ( 0x2f, 0x88, 0x3c, 0x46, 0x48, 0xe1, 0x31, 0xe8, 0x6d, 0xdf,
		 0x9d, 0xca, 0x0d, 0x74, 0xf3, 0x0c, 0xa1, 0xce, 0x6e, 0xfb ) );

/** SHA-1 Test 5.6 : Retried second call to Generate */
HMAC_DRBG_TEST_GENERATE ( sha1_generate_5_6, HMAC_DRBG_SHA1,
	additional_input_empty,
	EXPECT ( 0xf9, 0x39, 0xa5, 0xab, 0x08, 0xa3, 0x9f, 0x23, 0x10, 0x70,
		 0xb0, 0xd4, 0xc9, 0x6d, 0xc2, 0x37, 0x90, 0xba, 0x01, 0x53 ),
	EXPECT ( 0xce, 0x6d, 0x08, 0xb4, 0xae, 0x2c, 0xe3, 0x83, 0xfd, 0xab,
		 0xb0, 0x1e, 0xaa, 0xfc, 0x9c, 0x8e, 0x76, 0xa0, 0xd4, 0x72 ),
	EXPECT ( 0x84, 0xad, 0xd5, 0xe2, 0xd2, 0x04, 0x1c, 0x01, 0x72, 0x3a,
		 0x4d, 0xe4, 0x33, 0x5b, 0x13, 0xef, 0xdf, 0x16, 0xb0, 0xe5,
		 0x1a, 0x0a, 0xd3, 0x9b, 0xd1, 0x5e, 0x86, 0x2e, 0x64, 0x4f,
		 0x31, 0xe4, 0xa2, 0xd7, 0xd8, 0x43, 0xe5, 0x7c, 0x59, 0x68 ) );

/** SHA-1 Test 6 : Instantiate */
#define sha1_instantiate_6 sha1_instantiate_1

/** SHA-1 Test 6.1 : First call to Generate */
HMAC_DRBG_TEST_GENERATE_FAIL ( sha1_generate_fail_6_1, HMAC_DRBG_SHA1,
	additional_input_1, ( 320 / 8 ) );

/** SHA-1 Test 6.2 : Reseed */
HMAC_DRBG_TEST_RESEED ( sha1_reseed_6_2, HMAC_DRBG_SHA1,
	entropy_input_1, additional_input_1,
	EXPECT ( 0x52, 0x28, 0xa4, 0xb6, 0xa4, 0x46, 0x92, 0x90, 0x5e, 0xc0,
		 0x44, 0xbf, 0xf0, 0xbb, 0x4e, 0x25, 0xa3, 0x87, 0xca, 0xc1 ),
	EXPECT ( 0x24, 0x77, 0x32, 0xd0, 0x4c, 0xb8, 0x4e, 0xd4, 0x1a, 0xdd,
		 0x95, 0xa4, 0xb7, 0x8b, 0x50, 0xcd, 0x9b, 0x3d, 0x3f, 0x32 ) );

/** SHA-1 Test 6.3 : Retried first call to Generate */
HMAC_DRBG_TEST_GENERATE ( sha1_generate_6_3, HMAC_DRBG_SHA1,
	additional_input_empty,
	EXPECT ( 0xab, 0x3d, 0xd4, 0x89, 0x5b, 0xc8, 0xcd, 0x22, 0x71, 0xde,
		 0xba, 0x5f, 0x3c, 0x13, 0x63, 0x52, 0x6b, 0x8b, 0x74, 0x52 ),
	EXPECT ( 0xa8, 0x66, 0xc5, 0xef, 0xf2, 0xaf, 0x04, 0x2b, 0x11, 0x86,
		 0x44, 0x94, 0x45, 0x23, 0x7f, 0x9c, 0x02, 0x44, 0x98, 0x64 ),
	EXPECT ( 0xa1, 0xba, 0x8f, 0xa5, 0x8b, 0xb5, 0x01, 0x3f, 0x43, 0xf7,
		 0xb6, 0xed, 0x52, 0xb4, 0x53, 0x9f, 0xa1, 0x6d, 0xc7, 0x79,
		 0x57, 0xae, 0xe8, 0x15, 0xb9, 0xc0, 0x70, 0x04, 0xc7, 0xe9,
		 0x92, 0xeb, 0x8c, 0x7e, 0x59, 0x19, 0x64, 0xaf, 0xee, 0xa2 ) );

/** SHA-1 Test 6.4 : Second call to Generate */
HMAC_DRBG_TEST_GENERATE_FAIL ( sha1_generate_fail_6_4, HMAC_DRBG_SHA1,
	additional_input_2, ( 320 / 8 ) );

/** SHA-1 Test 6.5 : Reseed */
HMAC_DRBG_TEST_RESEED ( sha1_reseed_6_5, HMAC_DRBG_SHA1,
	entropy_input_2, additional_input_2,
	EXPECT ( 0xe5, 0x73, 0x9f, 0x9c, 0xf7, 0xff, 0x43, 0x84, 0xd1, 0x27,
		 0x3e, 0x02, 0x6b, 0x45, 0x31, 0x21, 0x36, 0x49, 0x4f, 0x41 ),
	EXPECT ( 0x30, 0xc3, 0x43, 0x05, 0xc2, 0xc6, 0x48, 0xb0, 0x57, 0xa6,
		 0x40, 0x22, 0x1b, 0x5c, 0x56, 0x57, 0x26, 0xcd, 0x32, 0xb2 ) );

/** SHA-1 Test 6.6 : Retried second call to Generate */
HMAC_DRBG_TEST_GENERATE ( sha1_generate_6_6, HMAC_DRBG_SHA1,
	additional_input_empty,
	EXPECT ( 0x61, 0x91, 0xca, 0x9b, 0xf0, 0x00, 0xd1, 0x0a, 0x71, 0x69,
		 0x0a, 0xc1, 0x0e, 0x09, 0xff, 0xc8, 0x92, 0xab, 0xde, 0x9a ),
	EXPECT ( 0x1e, 0xc0, 0x49, 0x0f, 0xa0, 0xb7, 0x65, 0x52, 0x7e, 0x5e,
		 0xa1, 0x8b, 0x53, 0x22, 0xb2, 0x8b, 0xdd, 0x0e, 0x7b, 0xc0 ),
	EXPECT ( 0x84, 0x26, 0x4a, 0x73, 0xa8, 0x18, 0xc9, 0x5c, 0x2f, 0x42,
		 0x4b, 0x37, 0xd3, 0xcc, 0x99, 0x0b, 0x04, 0x6f, 0xb5, 0x0c,
		 0x2d, 0xc6, 0x4a, 0x16, 0x42, 0x11, 0x88, 0x9a, 0x01, 0x0f,
		 0x24, 0x71, 0xa0, 0x91, 0x2f, 0xfe, 0xa1, 0xbf, 0x01, 0x95 ) );

/** SHA-1 Test 7 : Instantiation */
#define sha1_instantiate_7 sha1_instantiate_3

/** SHA-1 Test 7.1 : First call to Generate */
HMAC_DRBG_TEST_GENERATE_FAIL ( sha1_generate_fail_7_1, HMAC_DRBG_SHA1,
	additional_input_empty, ( 320 / 8 ) );

/** SHA-1 Test 7.2 : Reseed */
HMAC_DRBG_TEST_RESEED ( sha1_reseed_7_2, HMAC_DRBG_SHA1,
	entropy_input_1, additional_input_empty,
	EXPECT ( 0xb9, 0x25, 0x4d, 0x8a, 0xac, 0xba, 0x43, 0xfb, 0xda, 0xe6,
		 0x39, 0x4f, 0x2b, 0x3a, 0xfc, 0x5d, 0x58, 0x08, 0x00, 0xbf ),
	EXPECT ( 0x28, 0x40, 0x3b, 0x60, 0x36, 0x38, 0xd0, 0x7d, 0x79, 0x66,
		 0x66, 0x1e, 0xf6, 0x7b, 0x9d, 0x39, 0x05, 0xf4, 0x6d, 0xb9 ) );

/** SHA-1 Test 7.3 : Retried first call to Generate */
HMAC_DRBG_TEST_GENERATE ( sha1_generate_7_3, HMAC_DRBG_SHA1,
	additional_input_empty,
	EXPECT ( 0x64, 0xfe, 0x07, 0x4a, 0x6e, 0x77, 0x97, 0xd1, 0xa4, 0x35,
		 0xda, 0x89, 0x64, 0x48, 0x4d, 0x6c, 0xf8, 0xbd, 0xc0, 0x1b ),
	EXPECT ( 0x43, 0xe0, 0xc0, 0x52, 0x15, 0x86, 0xe9, 0x47, 0x3b, 0x06,
		 0x0d, 0x87, 0xd0, 0x8a, 0x23, 0x25, 0xfa, 0xe1, 0x49, 0xd1 ),
	EXPECT ( 0x6c, 0x37, 0xfd, 0xd7, 0x29, 0xaa, 0x40, 0xf8, 0x0b, 0xc6,
		 0xab, 0x08, 0xca, 0x7c, 0xc6, 0x49, 0x79, 0x4f, 0x69, 0x98,
		 0xb5, 0x70, 0x81, 0xe4, 0x22, 0x0f, 0x22, 0xc5, 0xc2, 0x83,
		 0xe2, 0xc9, 0x1b, 0x8e, 0x30, 0x5a, 0xb8, 0x69, 0xc6, 0x25 ) );

/** SHA-1 Test 7.4 : Second call to Generate */
HMAC_DRBG_TEST_GENERATE_FAIL ( sha1_generate_fail_7_4, HMAC_DRBG_SHA1,
	additional_input_empty, ( 320 / 8 ) );

/** SHA-1 Test 7.5 : Reseed */
HMAC_DRBG_TEST_RESEED ( sha1_reseed_7_5, HMAC_DRBG_SHA1,
	entropy_input_2, additional_input_empty,
	EXPECT ( 0x02, 0xbc, 0x57, 0x7f, 0xd1, 0x0e, 0xf7, 0x19, 0x3c, 0x1d,
		 0xb0, 0x98, 0xbd, 0x5b, 0x75, 0xc7, 0xc4, 0xb6, 0x79, 0x59 ),
	EXPECT ( 0xbc, 0xbd, 0xf0, 0x52, 0xe0, 0xe0, 0x2a, 0xe8, 0x9a, 0x77,
		 0x67, 0x94, 0x3f, 0x98, 0x65, 0xb8, 0xb7, 0x22, 0x90, 0x2d ) );

/** SHA-1 Test 7.6 : Retried second call to Generate */
HMAC_DRBG_TEST_GENERATE ( sha1_generate_7_6, HMAC_DRBG_SHA1,
	additional_input_empty,
	EXPECT ( 0x1a, 0xa4, 0x24, 0x1c, 0x69, 0x5e, 0x29, 0xc0, 0xa5, 0x9a,
		 0xd1, 0x8a, 0x60, 0x70, 0xe3, 0x38, 0xa5, 0x48, 0xbe, 0x92 ),
	EXPECT ( 0x03, 0x47, 0x35, 0x9b, 0xc9, 0xc7, 0xf8, 0x8c, 0xc8, 0x33,
		 0x0d, 0x4f, 0x59, 0xfb, 0xc7, 0x70, 0xb0, 0xb7, 0x7b, 0x03 ),
	EXPECT ( 0xca, 0xf5, 0x7d, 0xcf, 0xea, 0x39, 0x3b, 0x92, 0x36, 0xbf,
		 0x69, 0x1f, 0xa4, 0x56, 0xfe, 0xa7, 0xfd, 0xf1, 0xdf, 0x83,
		 0x61, 0x48, 0x2c, 0xa5, 0x4d, 0x5f, 0xa7, 0x23, 0xf4, 0xc8,
		 0x8b, 0x4f, 0xa5, 0x04, 0xbf, 0x03, 0x27, 0x7f, 0xa7, 0x83 ) );

/** SHA-1 Test 8 : Instantiate */
#define sha1_instantiate_8 sha1_instantiate_3

/** SHA-1 Test 8.1 : First call to Generate */
HMAC_DRBG_TEST_GENERATE_FAIL ( sha1_generate_fail_8_1, HMAC_DRBG_SHA1,
	additional_input_1, ( 320 / 8 ) );

/** SHA-1 Test 8.2 : Reseed */
HMAC_DRBG_TEST_RESEED ( sha1_reseed_8_2, HMAC_DRBG_SHA1,
	entropy_input_1, additional_input_1,
	EXPECT ( 0xc0, 0x95, 0x48, 0xc0, 0xd3, 0xc8, 0x61, 0xd7, 0x40, 0xf2,
		 0x83, 0x7d, 0x72, 0xb5, 0x07, 0x23, 0x5c, 0x26, 0xdb, 0x82 ),
	EXPECT ( 0x17, 0x4b, 0x3f, 0x84, 0xc3, 0x53, 0x1f, 0x7c, 0x0a, 0x2e,
		 0x54, 0x21, 0x23, 0x4e, 0xa1, 0x6b, 0x70, 0x8d, 0xdf, 0x0d ) );

/** SHA-1 Test 8.3 : Retried first call to Generate */
HMAC_DRBG_TEST_GENERATE ( sha1_generate_8_3, HMAC_DRBG_SHA1,
	additional_input_empty,
	EXPECT ( 0x60, 0x3f, 0x09, 0x49, 0x27, 0x9c, 0x70, 0xe8, 0xc6, 0x6c,
		 0x0f, 0x56, 0x37, 0xc0, 0xf3, 0x75, 0x60, 0x07, 0xe5, 0xac ),
	EXPECT ( 0xf2, 0xb3, 0x3b, 0x21, 0x15, 0x1f, 0xaf, 0x61, 0x20, 0x01,
		 0x83, 0x10, 0xf4, 0x4e, 0x4c, 0xd0, 0xbf, 0xe3, 0x68, 0xea ),
	EXPECT ( 0xbd, 0x07, 0xc2, 0x5c, 0xfd, 0x7c, 0x5e, 0x3a, 0x4e, 0xaa,
		 0x6e, 0x2e, 0xdc, 0x5a, 0xb7, 0xea, 0x49, 0x42, 0xa0, 0x91,
		 0x34, 0x71, 0xfd, 0xa5, 0x5c, 0x6d, 0xdd, 0x2c, 0x03, 0xef,
		 0xa3, 0xb9, 0x64, 0x3a, 0xb3, 0xbb, 0x22, 0xf6, 0xc9, 0xf2 ) );

/** SHA-1 Test 8.4 : Second call to Generate */
HMAC_DRBG_TEST_GENERATE_FAIL ( sha1_generate_fail_8_4, HMAC_DRBG_SHA1,
	additional_input_2, ( 320 / 8 ) );

/** SHA-1 Test 8.5 : Reseed */
HMAC_DRBG_TEST_RESEED ( sha1_reseed_8_5, HMAC_DRBG_SHA1,
	entropy_input_2, additional_input_2,
	EXPECT ( 0x89, 0x42, 0xa5, 0x4f, 0x34, 0x9e, 0x28, 0x1b, 0x84, 0xaa,
		 0x46, 0x95, 0x87, 0xfb, 0xdd, 0xaf, 0x9d, 0x11, 0x40, 0x82 ),
	EXPECT ( 0x07, 0x73, 0x0e, 0x3c, 0xbf, 0xfd, 0x3c, 0xaf, 0xd7, 0xa8,
		 0xaa, 0xe2, 0xbf, 0x01, 0xd6, 0x01, 0x43, 0x01, 0xe2, 0x4d ) );

/** SHA-1 Test 8.6 : Retried second call to Generate */
HMAC_DRBG_TEST_GENERATE ( sha1_generate_8_6, HMAC_DRBG_SHA1,
	additional_input_empty,
	EXPECT ( 0xbd, 0xe1, 0xb4, 0x6c, 0xdc, 0x54, 0x13, 0xb3, 0xd9, 0xf7,
		 0x35, 0xac, 0xdb, 0x80, 0xb1, 0x3c, 0x57, 0xbf, 0xe4, 0x73 ),
	EXPECT ( 0x72, 0x5a, 0x3c, 0x78, 0x20, 0xde, 0x1a, 0x06, 0xd0, 0x95,
		 0x81, 0x9c, 0xcf, 0x6f, 0x2c, 0x9b, 0x3a, 0x67, 0xf2, 0xce ),
	EXPECT ( 0xd1, 0xa9, 0xc1, 0xa2, 0x2c, 0x84, 0xfc, 0x23, 0xff, 0x22,
		 0x27, 0xef, 0x98, 0xec, 0x8b, 0xa9, 0xdf, 0x2a, 0x20, 0x9b,
		 0xa1, 0xdb, 0x09, 0x80, 0x9f, 0x57, 0xbf, 0xea, 0xe5, 0xb3,
		 0xe5, 0xf1, 0x46, 0xc7, 0x5f, 0x2d, 0x8d, 0xbb, 0x5e, 0x4a ) );

/** SHA-256 Test 1 : Instantiation */
HMAC_DRBG_TEST_INSTANTIATE ( sha256_instantiate_1, HMAC_DRBG_SHA256,
	entropy_input, nonce_sha256, personalisation_string_empty,
	EXPECT ( 0x3d, 0xda, 0x54, 0x3e, 0x7e, 0xef, 0x14, 0xf9, 0x36, 0x23,
		 0x7b, 0xe6, 0x5d, 0x09, 0x4b, 0x4d, 0xdc, 0x96, 0x9c, 0x0b,
		 0x2b, 0x5e, 0xaf, 0xb5, 0xd8, 0x05, 0xe8, 0x6c, 0xfa, 0x64,
		 0xd7, 0x41 ),
	EXPECT ( 0x2d, 0x02, 0xc2, 0xf8, 0x22, 0x51, 0x7d, 0x54, 0xb8, 0x17,
		 0x27, 0x9a, 0x59, 0x49, 0x1c, 0x41, 0xa1, 0x98, 0x9b, 0x3e,
		 0x38, 0x2d, 0xeb, 0xe8, 0x0d, 0x2c, 0x7f, 0x66, 0x0f, 0x44,
		 0x76, 0xc4 ) );

/** SHA-256 Test 1.1 : First call to Generate */
HMAC_DRBG_TEST_GENERATE ( sha256_generate_1_1, HMAC_DRBG_SHA256,
	additional_input_empty,
	EXPECT ( 0xdd, 0x30, 0x95, 0x79, 0x35, 0x38, 0x02, 0xcc, 0xdd, 0x43,
		 0x99, 0xc3, 0x69, 0x1c, 0x9d, 0xd9, 0x09, 0xdd, 0x3b, 0x2d,
		 0xd0, 0x03, 0xcc, 0xd5, 0x9d, 0x6f, 0x08, 0xd8, 0x5f, 0x2e,
		 0x35, 0x09 ),
	EXPECT ( 0xa1, 0xc2, 0x0f, 0xf2, 0x70, 0xa3, 0x9d, 0x2b, 0x8d, 0x03,
		 0xd6, 0x59, 0xb9, 0xdd, 0xd0, 0x11, 0xc2, 0xcc, 0xdf, 0x24,
		 0x48, 0x55, 0x7e, 0xf6, 0xa1, 0xa9, 0x15, 0xd1, 0x89, 0x40,
		 0xa6, 0x88 ),
	EXPECT ( 0xd6, 0x7b, 0x8c, 0x17, 0x34, 0xf4, 0x6f, 0xa3, 0xf7, 0x63,
		 0xcf, 0x57, 0xc6, 0xf9, 0xf4, 0xf2, 0xdc, 0x10, 0x89, 0xbd,
		 0x8b, 0xc1, 0xf6, 0xf0, 0x23, 0x95, 0x0b, 0xfc, 0x56, 0x17,
		 0x63, 0x52, 0x08, 0xc8, 0x50, 0x12, 0x38, 0xad, 0x7a, 0x44,
		 0x00, 0xde, 0xfe, 0xe4, 0x6c, 0x64, 0x0b, 0x61, 0xaf, 0x77,
		 0xc2, 0xd1, 0xa3, 0xbf, 0xaa, 0x90, 0xed, 0xe5, 0xd2, 0x07,
		 0x40, 0x6e, 0x54, 0x03 ) );

/** SHA-256 Test 1.2 : Second call to Generate */
HMAC_DRBG_TEST_GENERATE ( sha256_generate_1_2, HMAC_DRBG_SHA256,
	additional_input_empty,
	EXPECT ( 0x5c, 0xd5, 0xe5, 0x0a, 0x3e, 0x44, 0x8a, 0x07, 0xc3, 0xd2,
		 0xf2, 0xa3, 0xf9, 0xde, 0xbc, 0xc0, 0x46, 0x5f, 0x9c, 0xf1,
		 0x1c, 0xa1, 0x36, 0xe9, 0xb5, 0x04, 0xb4, 0xd3, 0x1c, 0x7f,
		 0xf1, 0xb8 ),
	EXPECT ( 0x33, 0xb3, 0x09, 0xf2, 0xff, 0x01, 0xce, 0x10, 0x4b, 0x44,
		 0x29, 0xb6, 0x75, 0xfa, 0xfa, 0x19, 0x01, 0x1e, 0x34, 0x8b,
		 0x28, 0x12, 0x71, 0x5a, 0x76, 0x37, 0xf6, 0xa6, 0xe6, 0x3b,
		 0x5d, 0x57 ),
	EXPECT ( 0x8f, 0xda, 0xec, 0x20, 0xf8, 0xb4, 0x21, 0x40, 0x70, 0x59,
		 0xe3, 0x58, 0x89, 0x20, 0xda, 0x7e, 0xda, 0x9d, 0xce, 0x3c,
		 0xf8, 0x27, 0x4d, 0xfa, 0x1c, 0x59, 0xc1, 0x08, 0xc1, 0xd0,
		 0xaa, 0x9b, 0x0f, 0xa3, 0x8d, 0xa5, 0xc7, 0x92, 0x03, 0x7c,
		 0x4d, 0x33, 0xcd, 0x07, 0x0c, 0xa7, 0xcd, 0x0c, 0x56, 0x08,
		 0xdb, 0xa8, 0xb8, 0x85, 0x65, 0x46, 0x39, 0xde, 0x21, 0x87,
		 0xb7, 0x4c, 0xb2, 0x63 ) );

/** SHA-256 Test 2 : Instantiation */
#define sha256_instantiate_2 sha256_instantiate_1

/** SHA-256 Test 2.1 : First call to Generate */
HMAC_DRBG_TEST_GENERATE ( sha256_generate_2_1, HMAC_DRBG_SHA256,
	additional_input_1,
	EXPECT ( 0x79, 0x1d, 0x31, 0x44, 0xb3, 0x02, 0xad, 0x6c, 0xe4, 0x32,
		 0x41, 0x34, 0x42, 0x10, 0xaa, 0xd0, 0xd3, 0x99, 0xed, 0xb7,
		 0xb5, 0x90, 0x6f, 0xb2, 0x51, 0xdb, 0x1c, 0xb6, 0x00, 0x04,
		 0xea, 0x51 ),
	EXPECT ( 0x58, 0xfd, 0x96, 0x5f, 0x4f, 0x99, 0x89, 0x3c, 0x17, 0xe6,
		 0xa3, 0x3c, 0xb8, 0xe9, 0x04, 0x15, 0xb5, 0x16, 0xd0, 0x06,
		 0x14, 0xa4, 0x49, 0xd4, 0x06, 0xe0, 0x3c, 0x68, 0x5b, 0xd8,
		 0x59, 0xbd ),
	EXPECT ( 0x41, 0x87, 0x87, 0x35, 0x81, 0x35, 0x41, 0x9b, 0x93, 0x81,
		 0x33, 0x53, 0x53, 0x06, 0x17, 0x6a, 0xfb, 0x25, 0x1c, 0xdd,
		 0x2b, 0xa3, 0x79, 0x88, 0x59, 0xb5, 0x66, 0xa0, 0x5c, 0xfb,
		 0x1d, 0x68, 0x0e, 0xa9, 0x25, 0x85, 0x6d, 0x5b, 0x84, 0xd5,
		 0x6a, 0xda, 0xe8, 0x70, 0x45, 0xa6, 0xba, 0x28, 0xd2, 0xc9,
		 0x08, 0xab, 0x75, 0xb7, 0xcc, 0x41, 0x43, 0x1f, 0xac, 0x59,
		 0xf3, 0x89, 0x18, 0xa3 ) );

/** SHA-256 Test 2.2 : Second call to Generate */
HMAC_DRBG_TEST_GENERATE ( sha256_generate_2_2, HMAC_DRBG_SHA256,
	additional_input_2,
	EXPECT ( 0xe7, 0x45, 0x8f, 0xb4, 0x4a, 0x36, 0x9a, 0x65, 0x3f, 0x2f,
		 0x8f, 0x57, 0x7b, 0xf9, 0x75, 0xc4, 0xb3, 0x62, 0xc4, 0xfe,
		 0x61, 0x8b, 0x2f, 0x1f, 0xf6, 0x76, 0x9b, 0x13, 0xc9, 0x4d,
		 0xec, 0xf4 ),
	EXPECT ( 0x19, 0x33, 0x4b, 0x8c, 0x31, 0xb7, 0x49, 0x32, 0xdd, 0xd7,
		 0xb2, 0xa4, 0x68, 0xf6, 0x43, 0x6d, 0xf9, 0x2e, 0x10, 0x0d,
		 0x39, 0xd3, 0xac, 0xb3, 0x68, 0xc7, 0x02, 0x9c, 0xb8, 0x83,
		 0xec, 0x89 ),
	EXPECT ( 0x7c, 0x06, 0x7b, 0xdd, 0xca, 0x81, 0x72, 0x48, 0x23, 0xd6,
		 0x4c, 0x69, 0x82, 0x92, 0x85, 0xbd, 0xbf, 0xf5, 0x37, 0x71,
		 0x61, 0x02, 0xc1, 0x88, 0x2e, 0x20, 0x22, 0x50, 0xe0, 0xfa,
		 0x5e, 0xf3, 0xa3, 0x84, 0xcd, 0x34, 0xa2, 0x0f, 0xfd, 0x1f,
		 0xbc, 0x91, 0xe0, 0xc5, 0x32, 0xa8, 0xa4, 0x21, 0xbc, 0x4a,
		 0xfe, 0x3c, 0xd4, 0x7f, 0x22, 0x32, 0x3e, 0xb4, 0xba, 0xe1,
		 0xa0, 0x07, 0x89, 0x81 ) );

/** SHA-256 Test 3 : Instantiation */
HMAC_DRBG_TEST_INSTANTIATE ( sha256_instantiate_3, HMAC_DRBG_SHA256,
	entropy_input, nonce_sha256, personalisation_string,
	EXPECT ( 0x65, 0x67, 0x3c, 0x34, 0x8e, 0x51, 0xcf, 0xac, 0xc4, 0x10,
		 0xbd, 0x20, 0x02, 0x49, 0xa5, 0x9a, 0x9d, 0x6b, 0xae, 0x77,
		 0x69, 0x04, 0x27, 0x1b, 0xb1, 0xf7, 0x18, 0xda, 0x1d, 0x18,
		 0x20, 0x42 ),
	EXPECT ( 0xe0, 0xf9, 0x1a, 0xc9, 0x96, 0x30, 0xee, 0xe6, 0x7c, 0xf8,
		 0x30, 0xcf, 0xd5, 0x04, 0x4f, 0xeb, 0xf5, 0x5c, 0x0c, 0x11,
		 0x50, 0x07, 0x99, 0x7a, 0xda, 0x11, 0x29, 0x6f, 0xc4, 0x16,
		 0x4a, 0x9a ) );

/** SHA-256 Test 3.1 : First call to Generate */
HMAC_DRBG_TEST_GENERATE ( sha256_generate_3_1, HMAC_DRBG_SHA256,
	additional_input_empty,
	EXPECT ( 0xf0, 0xb2, 0xf2, 0x42, 0xca, 0xd9, 0x92, 0xa7, 0x24, 0xf7,
		 0xe5, 0x59, 0x1d, 0x2f, 0x3b, 0x0c, 0x21, 0x57, 0xae, 0x70,
		 0xd5, 0x32, 0x78, 0x99, 0x40, 0xf1, 0x64, 0x45, 0x9b, 0x00,
		 0xc7, 0x49 ),
	EXPECT ( 0x1a, 0x03, 0xf9, 0x1c, 0x51, 0x20, 0xba, 0xca, 0x2b, 0xf6,
		 0xc6, 0x4d, 0xd7, 0x3a, 0xb1, 0x1d, 0xf6, 0xfd, 0x3f, 0xf1,
		 0xac, 0x3b, 0x57, 0x20, 0xa3, 0xf7, 0xfb, 0xe3, 0x9e, 0x7e,
		 0x7f, 0xe9 ),
	EXPECT ( 0x0d, 0xd9, 0xc8, 0x55, 0x89, 0xf3, 0x57, 0xc3, 0x89, 0xd6,
		 0xaf, 0x8d, 0xe9, 0xd7, 0x34, 0xa9, 0x17, 0xc7, 0x71, 0xef,
		 0x2d, 0x88, 0x16, 0xb9, 0x82, 0x59, 0x6e, 0xd1, 0x2d, 0xb4,
		 0x5d, 0x73, 0x4a, 0x62, 0x68, 0x08, 0x35, 0xc0, 0x2f, 0xda,
		 0x66, 0xb0, 0x8e, 0x1a, 0x36, 0x9a, 0xe2, 0x18, 0xf2, 0x6d,
		 0x52, 0x10, 0xad, 0x56, 0x42, 0x48, 0x87, 0x2d, 0x7a, 0x28,
		 0x78, 0x41, 0x59, 0xc3 ) );

/** SHA-256 Test 3.2 : Second call to Generate */
HMAC_DRBG_TEST_GENERATE ( sha256_generate_3_2, HMAC_DRBG_SHA256,
	additional_input_empty,
	EXPECT ( 0x5c, 0x0d, 0xec, 0x09, 0x37, 0x08, 0xc1, 0x7c, 0xa7, 0x6b,
		 0x57, 0xc0, 0xcb, 0x60, 0xcf, 0x88, 0x9d, 0xcc, 0x47, 0xad,
		 0x10, 0xbd, 0x64, 0xbc, 0x6a, 0x14, 0xb2, 0x3f, 0x20, 0x26,
		 0x07, 0x8a ),
	EXPECT ( 0x45, 0x67, 0x52, 0xa5, 0x11, 0xb8, 0x48, 0xbd, 0x05, 0xf1,
		 0x81, 0x9b, 0x9f, 0x6b, 0x15, 0x42, 0xc7, 0xd5, 0xec, 0xf9,
		 0x32, 0x73, 0x39, 0x26, 0x7a, 0x0c, 0x77, 0x23, 0x5b, 0x87,
		 0xdc, 0x5a ),
	EXPECT ( 0x46, 0xb4, 0xf4, 0x75, 0x6a, 0xe7, 0x15, 0xe0, 0xe5, 0x16,
		 0x81, 0xab, 0x29, 0x32, 0xde, 0x15, 0x23, 0xbe, 0x5d, 0x13,
		 0xba, 0xf0, 0xf4, 0x58, 0x8b, 0x11, 0xfe, 0x37, 0x2f, 0xda,
		 0x37, 0xab, 0xe3, 0x68, 0x31, 0x73, 0x41, 0xbc, 0x8b, 0xa9,
		 0x1f, 0xc5, 0xd8, 0x5b, 0x7f, 0xb8, 0xca, 0x8f, 0xbc, 0x30,
		 0x9a, 0x75, 0x8f, 0xd6, 0xfc, 0xa9, 0xdf, 0x43, 0xc7, 0x66,
		 0x0b, 0x22, 0x13, 0x22 ) );

/** SHA-256 Test 4 : Instantiation */
#define sha256_instantiate_4 sha256_instantiate_3

/** SHA-256 Test 4.1 : First call to Generate */
HMAC_DRBG_TEST_GENERATE ( sha256_generate_4_1, HMAC_DRBG_SHA256,
	additional_input_1,
	EXPECT ( 0x57, 0x2c, 0x03, 0x74, 0xc1, 0xa1, 0x01, 0x25, 0xbf, 0xa6,
		 0xae, 0xcd, 0x7c, 0xeb, 0xfe, 0x32, 0xf7, 0x52, 0xc3, 0xfb,
		 0x31, 0x67, 0x31, 0xb7, 0xcf, 0xdb, 0xde, 0xc2, 0x63, 0x56,
		 0x93, 0x2b ),
	EXPECT ( 0xd6, 0x8b, 0xf0, 0x41, 0xf3, 0xeb, 0x50, 0x88, 0x08, 0x8d,
		 0x8b, 0x8e, 0x71, 0x2c, 0x36, 0xae, 0x95, 0x83, 0xbb, 0x08,
		 0xfd, 0x1f, 0x90, 0x34, 0xa4, 0xe9, 0x42, 0xe9, 0xa6, 0x74,
		 0x7c, 0xe7 ),
	EXPECT ( 0x14, 0x78, 0xf2, 0x9e, 0x94, 0xb0, 0x2c, 0xb4, 0x0d, 0x3a,
		 0xab, 0x86, 0x24, 0x55, 0x57, 0xce, 0x13, 0xa8, 0xca, 0x2f,
		 0xdb, 0x65, 0x7d, 0x98, 0xef, 0xc1, 0x92, 0x34, 0x6b, 0x9f,
		 0xac, 0x33, 0xea, 0x58, 0xad, 0xa2, 0xcc, 0xa4, 0x32, 0xcc,
		 0xde, 0xfb, 0xcd, 0xaa, 0x8b, 0x82, 0xf5, 0x53, 0xef, 0x96,
		 0x61, 0x34, 0xe2, 0xcd, 0x13, 0x9f, 0x15, 0xf0, 0x1c, 0xad,
		 0x56, 0x85, 0x65, 0xa8 ) );

/** SHA-256 Test 4.2 : Second call to Generate */
HMAC_DRBG_TEST_GENERATE ( sha256_generate_4_2, HMAC_DRBG_SHA256,
	additional_input_2,
	EXPECT ( 0x28, 0x2e, 0x07, 0x34, 0x80, 0x80, 0x93, 0x75, 0x58, 0xb1,
		 0x39, 0x2e, 0x95, 0xab, 0x91, 0xe7, 0xc1, 0xf6, 0x22, 0xb2,
		 0x4f, 0xfb, 0x87, 0x20, 0xa5, 0xf0, 0xa5, 0xe0, 0x75, 0x50,
		 0xc7, 0xc2 ),
	EXPECT ( 0xdf, 0xc3, 0xbd, 0xb5, 0xf3, 0xbc, 0xf1, 0xaa, 0x68, 0x29,
		 0x8e, 0x79, 0x0d, 0x72, 0x0a, 0x67, 0xa7, 0x6e, 0x31, 0xb9,
		 0x2b, 0x9b, 0x35, 0xa8, 0xe5, 0x47, 0x1b, 0xb1, 0x7e, 0x30,
		 0x3c, 0x6b ),
	EXPECT ( 0x49, 0x7c, 0x7a, 0x16, 0xe8, 0x8a, 0x64, 0x11, 0xf8, 0xfc,
		 0xe1, 0x0e, 0xf5, 0x67, 0x63, 0xc6, 0x10, 0x25, 0x80, 0x1d,
		 0x8f, 0x51, 0xa7, 0x43, 0x52, 0xd6, 0x82, 0xcc, 0x23, 0xa0,
		 0xa8, 0xe6, 0x73, 0xca, 0xe0, 0x32, 0x28, 0x93, 0x90, 0x64,
		 0x7d, 0xc6, 0x83, 0xb7, 0x34, 0x28, 0x85, 0xd6, 0xb7, 0x6a,
		 0xb1, 0xda, 0x69, 0x6d, 0x3e, 0x97, 0xe2, 0x2d, 0xff, 0xdd,
		 0xff, 0xfd, 0x8d, 0xf0 ) );

/** SHA-256 Test 5 : Instantiation */
#define sha256_instantiate_5 sha256_instantiate_1

/** SHA-256 Test 5.1 : First call to Generate */
HMAC_DRBG_TEST_GENERATE_FAIL ( sha256_generate_fail_5_1, HMAC_DRBG_SHA256,
	additional_input_empty, ( 512 / 8 ) );

/** SHA-256 Test 5.2 : Reseed */
HMAC_DRBG_TEST_RESEED ( sha256_reseed_5_2, HMAC_DRBG_SHA256,
	entropy_input_1, additional_input_empty,
	EXPECT ( 0xb8, 0x40, 0x07, 0xe3, 0xe2, 0x7f, 0x34, 0xf9, 0xa7, 0x82,
		 0x0b, 0x7a, 0xb5, 0x9b, 0xbe, 0xfc, 0xd0, 0xc4, 0xac, 0xae,
		 0xde, 0x4b, 0x0b, 0x36, 0xb1, 0x47, 0xb8, 0x97, 0x79, 0xfd,
		 0x74, 0x9d ),
	EXPECT ( 0xa7, 0x2b, 0x8f, 0xee, 0x92, 0x39, 0x2f, 0x0a, 0x9d, 0x2d,
		 0x61, 0xbf, 0x09, 0xa4, 0xdf, 0xcc, 0x9d, 0xe6, 0x9a, 0x16,
		 0xa5, 0xf1, 0x50, 0x22, 0x4c, 0x3e, 0xf6, 0x04, 0x2d, 0x15,
		 0x21, 0xfc ) );

/** SHA-256 Test 5.3 : Retried first call to Generate */
HMAC_DRBG_TEST_GENERATE ( sha256_generate_5_3, HMAC_DRBG_SHA256,
	additional_input_empty,
	EXPECT ( 0x43, 0x48, 0xaf, 0x84, 0x20, 0x84, 0x2f, 0xa0, 0x77, 0xb9,
		 0xd3, 0xdb, 0xa8, 0xdc, 0xe9, 0xb3, 0xe1, 0xdf, 0x73, 0x4f,
		 0xfc, 0xe1, 0xbe, 0xa5, 0xb9, 0xe2, 0xb1, 0x54, 0xdc, 0x5e,
		 0xc6, 0x15 ),
	EXPECT ( 0xd2, 0xc1, 0xac, 0x27, 0x88, 0x5d, 0x43, 0x32, 0x76, 0x71,
		 0x31, 0x46, 0x32, 0xea, 0x60, 0x43, 0x3c, 0xca, 0x72, 0x73,
		 0x04, 0x56, 0x9e, 0xa7, 0xd4, 0x71, 0xfe, 0xa7, 0xdb, 0x7d,
		 0x31, 0x5d ),
	EXPECT ( 0xfa, 0xbd, 0x0a, 0xe2, 0x5c, 0x69, 0xdc, 0x2e, 0xfd, 0xef,
		 0xb7, 0xf2, 0x0c, 0x5a, 0x31, 0xb5, 0x7a, 0xc9, 0x38, 0xab,
		 0x77, 0x1a, 0xa1, 0x9b, 0xf8, 0xf5, 0xf1, 0x46, 0x8f, 0x66,
		 0x5c, 0x93, 0x8c, 0x9a, 0x1a, 0x5d, 0xf0, 0x62, 0x8a, 0x56,
		 0x90, 0xf1, 0x5a, 0x1a, 0xd8, 0xa6, 0x13, 0xf3, 0x1b, 0xbd,
		 0x65, 0xee, 0xad, 0x54, 0x57, 0xd5, 0xd2, 0x69, 0x47, 0xf2,
		 0x9f, 0xe9, 0x1a, 0xa7 ) );

/** SHA-256 Test 5.4 : Second call to Generate */
HMAC_DRBG_TEST_GENERATE_FAIL ( sha256_generate_fail_5_4, HMAC_DRBG_SHA256,
	additional_input_empty, ( 512 / 8 ) );

/** SHA-256 Test 5.5 : Reseed */
HMAC_DRBG_TEST_RESEED ( sha256_reseed_5_5, HMAC_DRBG_SHA256,
	entropy_input_2, additional_input_empty,
	EXPECT ( 0xbf, 0xa0, 0x2c, 0xe7, 0xe9, 0x2d, 0xe9, 0x2b, 0x18, 0x24,
		 0x28, 0x86, 0x89, 0x0e, 0x58, 0x6f, 0x83, 0x69, 0x06, 0xac,
		 0xe9, 0xe5, 0x54, 0xf1, 0xb0, 0xed, 0x63, 0x57, 0x3c, 0xb8,
		 0xb5, 0x03 ),
	EXPECT ( 0xd3, 0x24, 0x03, 0xee, 0xa9, 0xdc, 0xe1, 0x61, 0x6e, 0x4e,
		 0x11, 0x55, 0xb9, 0x23, 0xd8, 0x84, 0x2c, 0xc6, 0xe7, 0x84,
		 0xc6, 0x7a, 0x93, 0x85, 0xb2, 0xa6, 0x37, 0xf1, 0x02, 0xfa,
		 0x45, 0xd5 ) );

/** SHA-256 Test 5.6 : Retried second call to Generate */
HMAC_DRBG_TEST_GENERATE ( sha256_generate_5_6, HMAC_DRBG_SHA256,
	additional_input_empty,
	EXPECT ( 0x81, 0x21, 0xf7, 0x76, 0x4c, 0x08, 0x1e, 0xe9, 0xd1, 0x17,
		 0x1e, 0xd1, 0x87, 0xba, 0xe0, 0x88, 0x95, 0xca, 0xe2, 0x30,
		 0xd0, 0xa2, 0x5e, 0x37, 0x39, 0xc5, 0x7d, 0x54, 0x16, 0x10,
		 0x9b, 0x82 ),
	EXPECT ( 0x37, 0x84, 0x97, 0x7c, 0xc0, 0xe5, 0x9f, 0xbc, 0x9c, 0xda,
		 0x4e, 0x11, 0x92, 0x47, 0x5c, 0x6e, 0xfa, 0xf8, 0x07, 0x20,
		 0x19, 0x86, 0x21, 0x22, 0xcb, 0x6b, 0xce, 0xaa, 0xcc, 0x4a,
		 0x17, 0x5e ),
	EXPECT ( 0x6b, 0xd9, 0x25, 0xb0, 0xe1, 0xc2, 0x32, 0xef, 0xd6, 0x7c,
		 0xcd, 0x84, 0xf7, 0x22, 0xe9, 0x27, 0xec, 0xb4, 0x6a, 0xb2,
		 0xb7, 0x40, 0x01, 0x47, 0x77, 0xaf, 0x14, 0xba, 0x0b, 0xbf,
		 0x53, 0xa4, 0x5b, 0xdb, 0xb6, 0x2b, 0x3f, 0x7d, 0x0b, 0x9c,
		 0x8e, 0xea, 0xd0, 0x57, 0xc0, 0xec, 0x75, 0x4e, 0xf8, 0xb5,
		 0x3e, 0x60, 0xa1, 0xf4, 0x34, 0xf0, 0x59, 0x46, 0xa8, 0xb6,
		 0x86, 0xaf, 0xbc, 0x7a ) );

/** SHA-256 Test 6 : Instantiate */
#define sha256_instantiate_6 sha256_instantiate_1

/** SHA-256 Test 6.1 : First call to Generate */
HMAC_DRBG_TEST_GENERATE_FAIL ( sha256_generate_fail_6_1, HMAC_DRBG_SHA256,
	additional_input_1, ( 512 / 8 ) );

/** SHA-256 Test 6.2 : Reseed */
HMAC_DRBG_TEST_RESEED ( sha256_reseed_6_2, HMAC_DRBG_SHA256,
	entropy_input_1, additional_input_1,
	EXPECT ( 0xc1, 0x25, 0xea, 0x99, 0x75, 0x8e, 0xbb, 0x9a, 0x6f, 0x69,
		 0xae, 0x31, 0x2a, 0xc2, 0x04, 0xb5, 0x94, 0xc0, 0x0a, 0xb6,
		 0x8b, 0x81, 0x6e, 0x3a, 0x52, 0x12, 0x8e, 0x02, 0x78, 0xa5,
		 0x84, 0xac ),
	EXPECT ( 0xb2, 0xcb, 0x2b, 0x89, 0x12, 0x3f, 0x5b, 0x4a, 0xf5, 0x87,
		 0xb8, 0xf6, 0xbd, 0xc5, 0x42, 0x7a, 0x99, 0x14, 0x19, 0xd3,
		 0x53, 0x07, 0x7c, 0x68, 0x5e, 0x70, 0x7a, 0xcd, 0xf8, 0xe9,
		 0xfd, 0xa9 ) );

/** SHA-256 Test 6.3 : Retried first call to Generate */
HMAC_DRBG_TEST_GENERATE ( sha256_generate_6_3, HMAC_DRBG_SHA256,
	additional_input_empty,
	EXPECT ( 0xc6, 0xed, 0x8f, 0xed, 0x71, 0x57, 0xa4, 0xd0, 0x9e, 0xa1,
		 0xdd, 0xe8, 0x94, 0x6b, 0x54, 0x43, 0x3e, 0xcc, 0x54, 0x49,
		 0xa4, 0xa3, 0x52, 0xaf, 0x45, 0x76, 0x4e, 0xe6, 0x73, 0x4b,
		 0xbb, 0x04 ),
	EXPECT ( 0xeb, 0xc7, 0x75, 0x25, 0x6b, 0xb7, 0x81, 0x24, 0x1e, 0x9c,
		 0x70, 0xbb, 0xcf, 0x73, 0x2b, 0xdc, 0x90, 0xad, 0x10, 0xd9,
		 0xdd, 0x3a, 0x89, 0x6e, 0xcc, 0x12, 0xb9, 0x2f, 0xfb, 0x63,
		 0x45, 0xab ),
	EXPECT ( 0x08, 0x5d, 0x57, 0xaf, 0x6b, 0xab, 0xcf, 0x2b, 0x9a, 0xee,
		 0xf3, 0x87, 0xd5, 0x31, 0x65, 0x0e, 0x6a, 0x50, 0x5c, 0x54,
		 0x40, 0x6a, 0xb3, 0x7a, 0x52, 0x89, 0x9e, 0x0e, 0xca, 0xb3,
		 0x63, 0x2b, 0x7a, 0x06, 0x8a, 0x28, 0x14, 0xc6, 0xdf, 0x6a,
		 0xe5, 0x32, 0xb6, 0x58, 0xd0, 0xd9, 0x74, 0x1c, 0x84, 0x77,
		 0x5f, 0xee, 0x45, 0xb6, 0x84, 0xcd, 0xbd, 0xc2, 0x5f, 0xbc,
		 0xb4, 0xd8, 0xf3, 0x10 ) );

/** SHA-256 Test 6.4 : Second call to Generate */
HMAC_DRBG_TEST_GENERATE_FAIL ( sha256_generate_fail_6_4, HMAC_DRBG_SHA256,
	additional_input_2, ( 512 / 8 ) );

/** SHA-256 Test 6.5 : Reseed */
HMAC_DRBG_TEST_RESEED ( sha256_reseed_6_5, HMAC_DRBG_SHA256,
	entropy_input_2, additional_input_2,
	EXPECT ( 0xfc, 0x51, 0xda, 0x84, 0xf9, 0x69, 0x6b, 0xcc, 0x84, 0xc8,
		 0xf2, 0xac, 0xb9, 0x24, 0xbc, 0xdf, 0x72, 0xf8, 0x2e, 0xa2,
		 0xca, 0x64, 0x3f, 0x08, 0x3b, 0x0c, 0x16, 0xc3, 0x63, 0x4e,
		 0xfc, 0x62 ),
	EXPECT ( 0xb9, 0x74, 0xe4, 0x37, 0x0a, 0xd5, 0x76, 0xbb, 0x99, 0xc4,
		 0xe4, 0x9e, 0xa6, 0x80, 0xbf, 0xf9, 0x8d, 0xe9, 0xe1, 0x2f,
		 0xec, 0xd0, 0x13, 0xde, 0xd4, 0x3c, 0x80, 0xf6, 0x9a, 0x7a,
		 0xde, 0x8a ) );

/** SHA-256 Test 6.6 : Retried second call to Generate */
HMAC_DRBG_TEST_GENERATE ( sha256_generate_6_6, HMAC_DRBG_SHA256,
	additional_input_empty,
	EXPECT ( 0x56, 0xa2, 0xb4, 0x46, 0x32, 0xcb, 0x8f, 0xc3, 0xa6, 0x40,
		 0x09, 0xbf, 0xd6, 0xec, 0x95, 0xe5, 0x6c, 0xef, 0x8e, 0x7c,
		 0x91, 0x2a, 0xa8, 0x2b, 0x16, 0xf6, 0x14, 0x91, 0x5d, 0x9c,
		 0xd6, 0xe3 ),
	EXPECT ( 0xb5, 0xb3, 0x96, 0xa0, 0x15, 0x76, 0xb0, 0xfe, 0x42, 0xf4,
		 0x08, 0x44, 0x55, 0x6c, 0x4c, 0xf4, 0xb6, 0x80, 0x4c, 0x94,
		 0xde, 0x9d, 0x62, 0x38, 0xf1, 0xf7, 0xe7, 0xaf, 0x5c, 0x72,
		 0x57, 0xf3 ),
	EXPECT ( 0x9b, 0x21, 0x9f, 0xd9, 0x0d, 0xe2, 0xa0, 0x8e, 0x49, 0x34,
		 0x05, 0xcf, 0x87, 0x44, 0x17, 0xb5, 0x82, 0x67, 0x70, 0xf3,
		 0x94, 0x48, 0x15, 0x55, 0xdc, 0x66, 0x8a, 0xcd, 0x96, 0xb9,
		 0xa3, 0xe5, 0x6f, 0x9d, 0x2c, 0x32, 0x5e, 0x26, 0xd4, 0x7c,
		 0x1d, 0xfc, 0xfc, 0x8f, 0xbf, 0x86, 0x12, 0x6f, 0x40, 0xa1,
		 0xe6, 0x39, 0x60, 0xf6, 0x27, 0x49, 0x34, 0x2e, 0xcd, 0xb7,
		 0x1b, 0x24, 0x0d, 0xc6 ) );

/** SHA-256 Test 7 : Instantiation */
#define sha256_instantiate_7 sha256_instantiate_3

/** SHA-256 Test 7.1 : First call to Generate */
HMAC_DRBG_TEST_GENERATE_FAIL ( sha256_generate_fail_7_1, HMAC_DRBG_SHA256,
	additional_input_empty, ( 512 / 8 ) );

/** SHA-256 Test 7.2 : Reseed */
HMAC_DRBG_TEST_RESEED ( sha256_reseed_7_2, HMAC_DRBG_SHA256,
	entropy_input_1, additional_input_empty,
	EXPECT ( 0x44, 0x76, 0xc6, 0xd1, 0x1f, 0xc3, 0x5d, 0x44, 0x09, 0xd9,
		 0x03, 0x2e, 0x45, 0x3b, 0x0f, 0x0d, 0xc3, 0x31, 0x4d, 0xb8,
		 0x62, 0xcb, 0xdb, 0x60, 0x9c, 0x56, 0x02, 0x20, 0x8d, 0x4c,
		 0x88, 0xd8 ),
	EXPECT ( 0x95, 0xef, 0x78, 0x5a, 0x61, 0xc2, 0xf7, 0xb3, 0x6b, 0xc5,
		 0x96, 0xba, 0x4b, 0xa2, 0x08, 0xa5, 0x2c, 0x6d, 0xc2, 0x03,
		 0x63, 0x6d, 0x8f, 0x17, 0x87, 0x45, 0x3b, 0x85, 0x2b, 0x7e,
		 0x49, 0xec ) );

/** SHA-256 Test 7.3 : Retried first call to Generate */
HMAC_DRBG_TEST_GENERATE ( sha256_generate_7_3, HMAC_DRBG_SHA256,
	additional_input_empty,
	EXPECT ( 0x0d, 0xf9, 0x11, 0x0e, 0x2f, 0x22, 0x58, 0x98, 0x24, 0xa9,
		 0x47, 0x6c, 0x8e, 0x32, 0x08, 0x8e, 0x51, 0xa0, 0xda, 0x36,
		 0x63, 0x3f, 0x8c, 0xd1, 0xf7, 0x54, 0x7d, 0xff, 0x69, 0x6e,
		 0x4b, 0x29 ),
	EXPECT ( 0xc0, 0xe3, 0xc8, 0xed, 0x5a, 0x8b, 0x57, 0x9e, 0x3f, 0xef,
		 0x9d, 0xf3, 0xb7, 0xc2, 0xc2, 0x12, 0x98, 0x07, 0x17, 0xcc,
		 0x91, 0xae, 0x18, 0x66, 0x45, 0xfa, 0xbb, 0x2c, 0xc7, 0x84,
		 0xd5, 0xd7 ),
	EXPECT ( 0xd8, 0xb6, 0x71, 0x30, 0x71, 0x41, 0x94, 0xff, 0xe5, 0xb2,
		 0xa3, 0x5d, 0xbc, 0xd5, 0xe1, 0xa2, 0x99, 0x42, 0xad, 0x5c,
		 0x68, 0xf3, 0xde, 0xb9, 0x4a, 0xdd, 0x9e, 0x9e, 0xba, 0xd8,
		 0x60, 0x67, 0xed, 0xf0, 0x49, 0x15, 0xfb, 0x40, 0xc3, 0x91,
		 0xea, 0xe7, 0x0c, 0x65, 0x9e, 0xaa, 0xe7, 0xef, 0x11, 0xa3,
		 0xd4, 0x6a, 0x5b, 0x08, 0x5e, 0xdd, 0x90, 0xcc, 0x72, 0xce,
		 0xa9, 0x89, 0x21, 0x0b ) );

/** SHA-256 Test 7.4 : Second call to Generate */
HMAC_DRBG_TEST_GENERATE_FAIL ( sha256_generate_fail_7_4, HMAC_DRBG_SHA256,
	additional_input_empty, ( 512 / 8 ) );

/** SHA-256 Test 7.5 : Reseed */
HMAC_DRBG_TEST_RESEED ( sha256_reseed_7_5, HMAC_DRBG_SHA256,
	entropy_input_2, additional_input_empty,
	EXPECT ( 0x3d, 0x77, 0x63, 0xe5, 0x30, 0x3d, 0xb5, 0x4b, 0xe2, 0x05,
		 0x44, 0xa8, 0x1e, 0x9f, 0x00, 0xca, 0xdc, 0xfc, 0x1c, 0xb2,
		 0x8d, 0xec, 0xb9, 0xcf, 0xc6, 0x99, 0xf6, 0x1d, 0xba, 0xf8,
		 0x80, 0x21 ),
	EXPECT ( 0xfe, 0xbc, 0x02, 0x79, 0xb7, 0x71, 0x0d, 0xec, 0x5c, 0x06,
		 0x7e, 0xbe, 0xfa, 0x06, 0x8e, 0x4b, 0x59, 0x67, 0x49, 0x1b,
		 0x7e, 0xef, 0x94, 0x75, 0x83, 0x50, 0x6d, 0x04, 0x97, 0xce,
		 0x67, 0xba ) );

/** SHA-256 Test 7.6 : Retried second call to Generate */
HMAC_DRBG_TEST_GENERATE ( sha256_generate_7_6, HMAC_DRBG_SHA256,
	additional_input_empty,
	EXPECT ( 0x2d, 0x21, 0xac, 0x94, 0x99, 0x2f, 0xd8, 0x2b, 0x09, 0x80,
		 0xd3, 0xd5, 0x95, 0x51, 0xb9, 0xd0, 0x7c, 0x8d, 0x54, 0xb2,
		 0x52, 0xb6, 0x16, 0x28, 0x93, 0x44, 0xf8, 0xac, 0x86, 0x9e,
		 0xd3, 0x5b ),
	EXPECT ( 0x61, 0x0c, 0x34, 0xcd, 0xbf, 0x6f, 0x75, 0x33, 0x54, 0x7f,
		 0x23, 0x32, 0xea, 0xc5, 0x7e, 0xe3, 0x1e, 0x72, 0x4f, 0xb2,
		 0x92, 0x55, 0x56, 0x6b, 0x59, 0x78, 0x33, 0x16, 0x6c, 0xd0,
		 0x39, 0x9f ),
	EXPECT ( 0x8b, 0xba, 0x71, 0xc2, 0x58, 0x3f, 0x25, 0x30, 0xc2, 0x59,
		 0xc9, 0x07, 0x84, 0xa5, 0x9a, 0xc4, 0x4d, 0x1c, 0x80, 0x56,
		 0x91, 0x7c, 0xcf, 0x38, 0x87, 0x88, 0x10, 0x2d, 0x73, 0x82,
		 0x4c, 0x6c, 0x11, 0xd5, 0xd6, 0x3b, 0xe1, 0xf0, 0x10, 0x17,
		 0xd8, 0x84, 0xcd, 0x69, 0xd9, 0x33, 0x4b, 0x9e, 0xbc, 0x01,
		 0xe7, 0xbd, 0x8f, 0xdf, 0x2a, 0x8e, 0x52, 0x57, 0x22, 0x93,
		 0xdc, 0x21, 0xc0, 0xe1 ) );

/** SHA-256 Test 8 : Instantiate */
#define sha256_instantiate_8 sha256_instantiate_3

/** SHA-256 Test 8.1 : First call to Generate */
HMAC_DRBG_TEST_GENERATE_FAIL ( sha256_generate_fail_8_1, HMAC_DRBG_SHA256,
	additional_input_1, ( 512 / 8 ) );

/** SHA-256 Test 8.2 : Reseed */
HMAC_DRBG_TEST_RESEED ( sha256_reseed_8_2, HMAC_DRBG_SHA256,
	entropy_input_1, additional_input_1,
	EXPECT ( 0xb3, 0x81, 0x38, 0x8c, 0x1d, 0x7c, 0xfd, 0x56, 0x59, 0x30,
		 0x99, 0x3b, 0xd9, 0x26, 0x90, 0x66, 0x50, 0x88, 0xd9, 0xb8,
		 0x39, 0x96, 0x9b, 0x87, 0xf1, 0x6d, 0xb6, 0xdf, 0x4e, 0x43,
		 0x00, 0xd7 ),
	EXPECT ( 0xfa, 0x04, 0x25, 0x64, 0x00, 0xe3, 0x42, 0xe6, 0x55, 0xf4,
		 0x33, 0x26, 0x94, 0xe3, 0xb2, 0x4c, 0x04, 0xfb, 0x85, 0xbf,
		 0x87, 0x80, 0x21, 0xe4, 0x52, 0xe7, 0x3b, 0x8f, 0x46, 0xd4,
		 0xbd, 0xc6 ) );

/** SHA-256 Test 8.3 : Retried first call to Generate */
HMAC_DRBG_TEST_GENERATE ( sha256_generate_8_3, HMAC_DRBG_SHA256,
	additional_input_empty,
	EXPECT ( 0xd4, 0x1f, 0x6f, 0x33, 0x65, 0x82, 0x21, 0x70, 0x50, 0xb1,
		 0xf6, 0x59, 0x28, 0xfd, 0x6e, 0x94, 0xcb, 0xc9, 0x45, 0x68,
		 0xfe, 0x3b, 0x6b, 0x53, 0x38, 0x9e, 0x1e, 0x3a, 0x5b, 0x49,
		 0xe1, 0x01 ),
	EXPECT ( 0xa6, 0x55, 0xc9, 0xe7, 0xd1, 0x33, 0xf1, 0xcd, 0x8b, 0x11,
		 0x61, 0xf2, 0x7d, 0x54, 0xe7, 0x5a, 0x7e, 0x7c, 0x80, 0x42,
		 0xbf, 0x74, 0xd4, 0x7f, 0x9f, 0xfd, 0x60, 0xe2, 0x45, 0xeb,
		 0xa5, 0x7e ),
	EXPECT ( 0x44, 0xd7, 0x8b, 0xbc, 0x3e, 0xb6, 0x7c, 0x59, 0xc2, 0x2f,
		 0x6c, 0x31, 0x00, 0x3d, 0x21, 0x2a, 0x78, 0x37, 0xcc, 0xd8,
		 0x4c, 0x43, 0x8b, 0x55, 0x15, 0x0f, 0xd0, 0x13, 0xa8, 0xa7,
		 0x8f, 0xe8, 0xed, 0xea, 0x81, 0xc6, 0x72, 0xe4, 0xb8, 0xdd,
		 0xc8, 0x18, 0x38, 0x86, 0xe6, 0x9c, 0x2e, 0x17, 0x7d, 0xf5,
		 0x74, 0xc1, 0xf1, 0x90, 0xdf, 0x27, 0x18, 0x50, 0xf8, 0xce,
		 0x55, 0xef, 0x20, 0xb8 ) );

/** SHA-256 Test 8.4 : Second call to Generate */
HMAC_DRBG_TEST_GENERATE_FAIL ( sha256_generate_fail_8_4, HMAC_DRBG_SHA256,
	additional_input_2, ( 512 / 8 ) );

/** SHA-256 Test 8.5 : Reseed */
HMAC_DRBG_TEST_RESEED ( sha256_reseed_8_5, HMAC_DRBG_SHA256,
	entropy_input_2, additional_input_2,
	EXPECT ( 0xfb, 0xa8, 0x05, 0x45, 0x3e, 0x3c, 0x9a, 0x73, 0x64, 0x58,
		 0x5c, 0xed, 0xbc, 0xd2, 0x92, 0x30, 0xfb, 0xc9, 0x3d, 0x6f,
		 0x12, 0x9d, 0x21, 0xed, 0xdd, 0xf6, 0x61, 0x3b, 0x3a, 0x8f,
		 0xf2, 0x83 ),
	EXPECT ( 0x83, 0x64, 0x7a, 0x33, 0x8c, 0x15, 0x3c, 0xba, 0xf0, 0xe4,
		 0x9a, 0x54, 0xa4, 0x4f, 0xea, 0x66, 0x70, 0xcf, 0xd7, 0xc1,
		 0x71, 0x4d, 0x4a, 0xb3, 0x5f, 0x11, 0x12, 0x3d, 0xf2, 0x7b,
		 0x69, 0xcf ) );

/** SHA-256 Test 8.6 : Retried second call to Generate */
HMAC_DRBG_TEST_GENERATE ( sha256_generate_8_6, HMAC_DRBG_SHA256,
	additional_input_empty,
	EXPECT ( 0xae, 0x59, 0xc7, 0x0a, 0x7c, 0x60, 0xed, 0x49, 0x83, 0x78,
		 0xea, 0x84, 0x5b, 0xe9, 0x7d, 0x8f, 0xf8, 0x81, 0xe0, 0xea,
		 0x37, 0x2e, 0x26, 0x5f, 0xa6, 0x72, 0x84, 0x29, 0x3e, 0x1a,
		 0x46, 0xac ),
	EXPECT ( 0xe2, 0xf0, 0x4d, 0xe3, 0xce, 0x21, 0x79, 0x61, 0xae, 0x2b,
		 0x2d, 0x20, 0xa7, 0xba, 0x7c, 0x6c, 0x82, 0x0b, 0x5b, 0x14,
		 0x92, 0x6e, 0x59, 0x56, 0xae, 0x6d, 0xfa, 0x2e, 0xd1, 0xd6,
		 0x39, 0x93 ),
	EXPECT ( 0x91, 0x77, 0x80, 0xdc, 0x0c, 0xe9, 0x98, 0x9f, 0xee, 0x6c,
		 0x08, 0x06, 0xd6, 0xda, 0x12, 0x3a, 0x18, 0x25, 0x29, 0x47,
		 0x58, 0xd4, 0xe1, 0xb5, 0x82, 0x68, 0x72, 0x31, 0x78, 0x0a,
		 0x2a, 0x9c, 0x33, 0xf1, 0xd1, 0x56, 0xcc, 0xad, 0x32, 0x77,
		 0x64, 0xb2, 0x9a, 0x4c, 0xb2, 0x69, 0x01, 0x77, 0xae, 0x96,
		 0xef, 0x9e, 0xe9, 0x2a, 0xd0, 0xc3, 0x40, 0xba, 0x0f, 0xd1,
		 0x20, 0x3c, 0x02, 0xc6 ) );

/**
 * Force a "reseed required" state
 *
 * @v state		HMAC_DRBG internal state
 */
static inline void force_reseed_required ( struct hmac_drbg_state *state ) {
	state->reseed_counter = ( HMAC_DRBG_RESEED_INTERVAL + 1 );
}

/**
 * Perform HMAC_DRBG self-test
 *
 */
static void hmac_drbg_test_exec ( void ) {
	struct hmac_drbg_state state;

	/*
	 * IMPORTANT NOTE
	 *
	 * The NIST test vector set includes several calls to
	 * HMAC_DRBG_Generate() that are expected to fail with a
	 * status of "Reseed required".  The pattern seems to be that
	 * when prediction resistance is requested, any call to
	 * HMAC_DRBG_Generate() is at first expected to fail.  After
	 * an explicit reseeding, the call to HMAC_DRBG_Generate() is
	 * retried, and on this second time it is expected to succeed.
	 *
	 * This pattern does not match the specifications for
	 * HMAC_DRBG_Generate(): neither HMAC_DRBG_Generate_algorithm
	 * (defined in ANS X9.82 Part 3-2007 Section 10.2.2.2.5 (NIST
	 * SP 800-90 Section 10.1.2.5)) nor the higher-level wrapper
	 * Generate_function defined in ANS X9.82 Part 3-2007 Section
	 * 9.4 (NIST SP 800-90 Section 9.3)) can possibly exhibit this
	 * behaviour:
	 *
	 * a) HMAC_DRBG_Generate_algorithm can return a "reseed
	 *    required" status only as a result of the test
	 *
	 *      "1. If reseed_counter > reseed_interval, then return
	 *       an indication that a reseed is required."
	 *
	 *    Since the reseed interval is independent of any request
	 *    for prediction resistance, and since the reseed interval
	 *    is not specified as part of the NIST test vector set,
	 *    then this cannot be the source of the "Reseed required"
	 *    failure expected by the NIST test vector set.
	 *
	 * b) Generate_function cannot return a "reseed required"
	 *    status under any circumstances.  If the underlying
	 *    HMAC_DRBG_Generate_algorithm call returns "reseed
	 *    required", then Generate_function will automatically
	 *    reseed and try again.
	 *
	 * To produce the behaviour expected by the NIST test vector
	 * set, we therefore contrive to produce a "reseed required"
	 * state where necessary by setting the reseed_counter to
	 * greater than the reseed_interval.
	 */

	/* SHA-1 Test 1 */
	instantiate_ok ( &state, &sha1_instantiate_1 );
	generate_ok ( &state, &sha1_generate_1_1 );
	generate_ok ( &state, &sha1_generate_1_2 );

	/* SHA-1 Test 2 */
	instantiate_ok ( &state, &sha1_instantiate_2 );
	generate_ok ( &state, &sha1_generate_2_1 );
	generate_ok ( &state, &sha1_generate_2_2 );

	/* SHA-1 Test 3 */
	instantiate_ok ( &state, &sha1_instantiate_3 );
	generate_ok ( &state, &sha1_generate_3_1 );
	generate_ok ( &state, &sha1_generate_3_2 );

	/* SHA-1 Test 4 */
	instantiate_ok ( &state, &sha1_instantiate_4 );
	generate_ok ( &state, &sha1_generate_4_1 );
	generate_ok ( &state, &sha1_generate_4_2 );

	/* SHA-1 Test 5 */
	instantiate_ok ( &state, &sha1_instantiate_5 );
	force_reseed_required ( &state ); /* See above comments */
	generate_fail_ok ( &state, &sha1_generate_fail_5_1 );
	reseed_ok ( &state, &sha1_reseed_5_2 );
	generate_ok ( &state, &sha1_generate_5_3 );
	force_reseed_required ( &state ); /* See above comments */
	generate_fail_ok ( &state, &sha1_generate_fail_5_4 );
	reseed_ok ( &state, &sha1_reseed_5_5 );
	generate_ok ( &state, &sha1_generate_5_6 );

	/* SHA-1 Test 6 */
	instantiate_ok ( &state, &sha1_instantiate_6 );
	force_reseed_required ( &state ); /* See above comments */
	generate_fail_ok ( &state, &sha1_generate_fail_6_1 );
	reseed_ok ( &state, &sha1_reseed_6_2 );
	generate_ok ( &state, &sha1_generate_6_3 );
	force_reseed_required ( &state ); /* See above comments */
	generate_fail_ok ( &state, &sha1_generate_fail_6_4 );
	reseed_ok ( &state, &sha1_reseed_6_5 );
	generate_ok ( &state, &sha1_generate_6_6 );

	/* SHA-1 Test 7 */
	instantiate_ok ( &state, &sha1_instantiate_7 );
	force_reseed_required ( &state ); /* See above comments */
	generate_fail_ok ( &state, &sha1_generate_fail_7_1 );
	reseed_ok ( &state, &sha1_reseed_7_2 );
	generate_ok ( &state, &sha1_generate_7_3 );
	force_reseed_required ( &state ); /* See above comments */
	generate_fail_ok ( &state, &sha1_generate_fail_7_4 );
	reseed_ok ( &state, &sha1_reseed_7_5 );
	generate_ok ( &state, &sha1_generate_7_6 );

	/* SHA-1 Test 8 */
	instantiate_ok ( &state, &sha1_instantiate_8 );
	force_reseed_required ( &state ); /* See above comments */
	generate_fail_ok ( &state, &sha1_generate_fail_8_1 );
	reseed_ok ( &state, &sha1_reseed_8_2 );
	generate_ok ( &state, &sha1_generate_8_3 );
	force_reseed_required ( &state ); /* See above comments */
	generate_fail_ok ( &state, &sha1_generate_fail_8_4 );
	reseed_ok ( &state, &sha1_reseed_8_5 );
	generate_ok ( &state, &sha1_generate_8_6 );

	/* SHA-256 Test 1 */
	instantiate_ok ( &state, &sha256_instantiate_1 );
	generate_ok ( &state, &sha256_generate_1_1 );
	generate_ok ( &state, &sha256_generate_1_2 );

	/* SHA-256 Test 2 */
	instantiate_ok ( &state, &sha256_instantiate_2 );
	generate_ok ( &state, &sha256_generate_2_1 );
	generate_ok ( &state, &sha256_generate_2_2 );

	/* SHA-256 Test 3 */
	instantiate_ok ( &state, &sha256_instantiate_3 );
	generate_ok ( &state, &sha256_generate_3_1 );
	generate_ok ( &state, &sha256_generate_3_2 );

	/* SHA-256 Test 4 */
	instantiate_ok ( &state, &sha256_instantiate_4 );
	generate_ok ( &state, &sha256_generate_4_1 );
	generate_ok ( &state, &sha256_generate_4_2 );

	/* SHA-256 Test 5 */
	instantiate_ok ( &state, &sha256_instantiate_5 );
	force_reseed_required ( &state ); /* See above comments */
	generate_fail_ok ( &state, &sha256_generate_fail_5_1 );
	reseed_ok ( &state, &sha256_reseed_5_2 );
	generate_ok ( &state, &sha256_generate_5_3 );
	force_reseed_required ( &state ); /* See above comments */
	generate_fail_ok ( &state, &sha256_generate_fail_5_4 );
	reseed_ok ( &state, &sha256_reseed_5_5 );
	generate_ok ( &state, &sha256_generate_5_6 );

	/* SHA-256 Test 6 */
	instantiate_ok ( &state, &sha256_instantiate_6 );
	force_reseed_required ( &state ); /* See above comments */
	generate_fail_ok ( &state, &sha256_generate_fail_6_1 );
	reseed_ok ( &state, &sha256_reseed_6_2 );
	generate_ok ( &state, &sha256_generate_6_3 );
	force_reseed_required ( &state ); /* See above comments */
	generate_fail_ok ( &state, &sha256_generate_fail_6_4 );
	reseed_ok ( &state, &sha256_reseed_6_5 );
	generate_ok ( &state, &sha256_generate_6_6 );

	/* SHA-256 Test 7 */
	instantiate_ok ( &state, &sha256_instantiate_7 );
	force_reseed_required ( &state ); /* See above comments */
	generate_fail_ok ( &state, &sha256_generate_fail_7_1 );
	reseed_ok ( &state, &sha256_reseed_7_2 );
	generate_ok ( &state, &sha256_generate_7_3 );
	force_reseed_required ( &state ); /* See above comments */
	generate_fail_ok ( &state, &sha256_generate_fail_7_4 );
	reseed_ok ( &state, &sha256_reseed_7_5 );
	generate_ok ( &state, &sha256_generate_7_6 );

	/* SHA-256 Test 8 */
	instantiate_ok ( &state, &sha256_instantiate_8 );
	force_reseed_required ( &state ); /* See above comments */
	generate_fail_ok ( &state, &sha256_generate_fail_8_1 );
	reseed_ok ( &state, &sha256_reseed_8_2 );
	generate_ok ( &state, &sha256_generate_8_3 );
	force_reseed_required ( &state ); /* See above comments */
	generate_fail_ok ( &state, &sha256_generate_fail_8_4 );
	reseed_ok ( &state, &sha256_reseed_8_5 );
	generate_ok ( &state, &sha256_generate_8_6 );
}

/** HMAC_DRBG self-test */
struct self_test hmac_drbg_test __self_test = {
	.name = "hmac_drbg",
	.exec = hmac_drbg_test_exec,
};
